import { HttpException, HttpStatus } from '@nestjs/common';
import { ZodError } from 'zod';
import { ExceptionType, PrimeException } from '../exceptions/prime.exception';
import { HttpExceptionFilter } from './http-exception.filter';

describe('HttpExceptionFilter', () => {
  let filter: HttpExceptionFilter;

  beforeEach(() => {
    filter = new HttpExceptionFilter();
  });

  it('should catch a generic HttpException and return the correct response', () => {
    const exception = new HttpException(
      'Test Exception',
      HttpStatus.BAD_REQUEST,
    );

    const mockResponse = {
      status: jest.fn().mockReturnThis(),
      json: jest.fn(),
    };

    const host = {
      switchToHttp: () => ({
        getResponse: () => mockResponse,
        getRequest: () => ({ url: '/test' }),
      }),
    };

    filter.catch(exception, host as any);

    expect(mockResponse.status).toHaveBeenCalledWith(HttpStatus.BAD_REQUEST);
    expect(mockResponse.json).toHaveBeenCalledWith({
      status: HttpStatus.BAD_REQUEST,
      message: 'Test Exception',
      type: 'INTERNAL',
    });
  });

  it('should correctly extract message when response.message is an array', () => {
    const exception = new HttpException(
      { message: ['Test', 'Exception'] },
      HttpStatus.BAD_REQUEST,
    );

    const mockResponse = {
      status: jest.fn().mockReturnThis(),
      json: jest.fn(),
    };

    const host = {
      switchToHttp: () => ({
        getResponse: () => mockResponse,
        getRequest: () => ({ url: '/test' }),
      }),
    };

    filter.catch(exception, host as any);

    expect(mockResponse.status).toHaveBeenCalledWith(HttpStatus.BAD_REQUEST);
    expect(mockResponse.json).toHaveBeenCalledWith({
      status: HttpStatus.BAD_REQUEST,
      message: 'Test, Exception',
      type: 'INTERNAL',
    });
  });

  it('should correctly extract message when response.message is an object', () => {
    const exception = new HttpException(
      { message: { message: 'Test Exception' } },
      HttpStatus.BAD_REQUEST,
    );

    const mockResponse = {
      status: jest.fn().mockReturnThis(),
      json: jest.fn(),
    };

    const host = {
      switchToHttp: () => ({
        getResponse: () => mockResponse,
        getRequest: () => ({ url: '/test' }),
      }),
    };

    filter.catch(exception, host as any);

    expect(mockResponse.status).toHaveBeenCalledWith(HttpStatus.BAD_REQUEST);
    expect(mockResponse.json).toHaveBeenCalledWith({
      status: HttpStatus.BAD_REQUEST,
      message: 'Test Exception',
      type: 'INTERNAL',
    });
  });

  it('should return default message when extractMessage returns undefined', () => {
    const exception = new HttpException(
      { message: {} },
      HttpStatus.INTERNAL_SERVER_ERROR,
    );

    const mockResponse = {
      status: jest.fn().mockReturnThis(),
      json: jest.fn(),
    };

    const host = {
      switchToHttp: () => ({
        getResponse: () => mockResponse,
        getRequest: () => ({ url: '/test' }),
      }),
    };

    filter.catch(exception, host as any);

    expect(mockResponse.status).toHaveBeenCalledWith(
      HttpStatus.INTERNAL_SERVER_ERROR,
    );
    expect(mockResponse.json).toHaveBeenCalledWith({
      status: HttpStatus.INTERNAL_SERVER_ERROR,
      message: 'Internal server error',
      type: 'INTERNAL',
    });
  });

  it('should handle AxiosError correctly', () => {
    const exception = {
      isAxiosError: true,
      message: 'Test AxiosError',
      response: {
        status: HttpStatus.BAD_REQUEST,
        statusText: 'Bad Request',
        headers: {},
        data: {},
      },
      config: {
        url: '/test',
        method: 'GET',
        data: {},
      },
    };

    const mockResponse = {
      status: jest.fn().mockReturnThis(),
      json: jest.fn(),
    };

    const host = {
      switchToHttp: () => ({
        getResponse: () => mockResponse,
        getRequest: () => ({ url: '/test' }),
      }),
    };

    filter.catch(exception, host as any);

    expect(mockResponse.status).toHaveBeenCalledWith(HttpStatus.BAD_REQUEST);
    expect(mockResponse.json).toHaveBeenCalledWith({
      status: HttpStatus.BAD_REQUEST,
      message: 'Test AxiosError',
      type: 'INTERNAL',
    });
  });

  it('should handle generic error correctly', () => {
    const exception = new Error('Test Error');

    const mockResponse = {
      status: jest.fn().mockReturnThis(),
      json: jest.fn(),
    };

    const host = {
      switchToHttp: () => ({
        getResponse: () => mockResponse,
        getRequest: () => ({ url: '/test' }),
      }),
    };

    filter.catch(exception, host as any);

    expect(mockResponse.status).toHaveBeenCalledWith(
      HttpStatus.INTERNAL_SERVER_ERROR,
    );
    expect(mockResponse.json).toHaveBeenCalledWith({
      status: HttpStatus.INTERNAL_SERVER_ERROR,
      message: 'Internal server error',
      type: 'INTERNAL',
    });
  });

  describe('ModeException', () => {
    it('should catch a ModeException and return the correct response', () => {
      const exception = new PrimeException('Test ModeException');

      const mockResponse = {
        status: jest.fn().mockReturnThis(),
        json: jest.fn(),
      };

      const host = {
        switchToHttp: () => ({
          getResponse: () => mockResponse,
          getRequest: () => ({ url: '/test' }),
        }),
      };

      filter.catch(exception, host as any);

      expect(mockResponse.status).toHaveBeenCalledWith(exception.getStatus());
      expect(mockResponse.json).toHaveBeenCalledWith({
        status: exception.getStatus(),
        message: exception.message,
        type: ExceptionType.BUSINESS,
      });
    });

    it('should catch a ModeException with a custom status code and return the correct response', () => {
      const exception = new PrimeException(
        'Test ModeException',
        HttpStatus.FORBIDDEN,
      );

      const mockResponse = {
        status: jest.fn().mockReturnThis(),
        json: jest.fn(),
      };

      const host = {
        switchToHttp: () => ({
          getResponse: () => mockResponse,
          getRequest: () => ({ url: '/test' }),
        }),
      };

      filter.catch(exception, host as any);

      expect(mockResponse.status).toHaveBeenCalledWith(HttpStatus.FORBIDDEN);
      expect(mockResponse.json).toHaveBeenCalledWith({
        status: HttpStatus.FORBIDDEN,
        message: exception.message,
        type: ExceptionType.BUSINESS,
      });
    });

    it('should catch a ModeException with a custom status code and type and return the correct response', () => {
      const exception = new PrimeException(
        'Test ModeException',
        HttpStatus.FORBIDDEN,
        ExceptionType.INTERNAL,
      );

      const mockResponse = {
        status: jest.fn().mockReturnThis(),
        json: jest.fn(),
      };

      const host = {
        switchToHttp: () => ({
          getResponse: () => mockResponse,
          getRequest: () => ({ url: '/test' }),
        }),
      };

      filter.catch(exception, host as any);

      expect(mockResponse.status).toHaveBeenCalledWith(HttpStatus.FORBIDDEN);
      expect(mockResponse.json).toHaveBeenCalledWith({
        status: HttpStatus.FORBIDDEN,
        message: exception.message,
        type: ExceptionType.INTERNAL,
      });
    });
  });

  it('should get null when getExceptionConstructorName is called with null', () => {
    const mockResponse = {
      status: jest.fn().mockReturnThis(),
      json: jest.fn(),
    };

    const host = {
      switchToHttp: () => ({
        getResponse: () => mockResponse,
        getRequest: () => ({ url: '/test' }),
      }),
    };

    filter.catch(undefined, host as any);

    expect(mockResponse.status).toHaveBeenCalledWith(
      HttpStatus.INTERNAL_SERVER_ERROR,
    );
    expect(mockResponse.json).toHaveBeenCalledWith({
      status: HttpStatus.INTERNAL_SERVER_ERROR,
      message: 'Internal server error',
      type: ExceptionType.INTERNAL,
    });
  });

  describe(`ZodError`, () => {
    it('should handle ZodError correctly', () => {
      const exception = new ZodError([
        {
          message: 'Test ZodError',
          code: 'invalid_type',
          path: ['test'],
          expected: 'string',
          received: 'number',
        },
      ]);

      const mockResponse = {
        status: jest.fn().mockReturnThis(),
        json: jest.fn(),
      };

      const host = {
        switchToHttp: () => ({
          getResponse: () => mockResponse,
          getRequest: () => ({ url: '/test' }),
        }),
      };

      filter.catch(exception, host as any);

      expect(mockResponse.status).toHaveBeenCalledWith(HttpStatus.BAD_REQUEST);
      expect(mockResponse.json).toHaveBeenCalledWith({
        status: HttpStatus.BAD_REQUEST,
        message: 'Validation error',
        type: ExceptionType.INTERNAL,
      });
    });

    it('should handle ZodError with an empty array of issues correctly', () => {
      const exception = new ZodError([]);

      const mockResponse = {
        status: jest.fn().mockReturnThis(),
        json: jest.fn(),
      };

      const host = {
        switchToHttp: () => ({
          getResponse: () => mockResponse,
          getRequest: () => ({ url: '/test' }),
        }),
      };

      filter.catch(exception, host as any);

      expect(mockResponse.status).toHaveBeenCalledWith(HttpStatus.BAD_REQUEST);
      expect(mockResponse.json).toHaveBeenCalledWith({
        status: HttpStatus.BAD_REQUEST,
        message: 'Validation error',
        type: ExceptionType.INTERNAL,
      });
    });

    it('should handle ZodError with an empty array of path', () => {
      const exception = new ZodError([
        {
          message: 'Test ZodError',
          code: 'invalid_type',
          path: [],
          expected: 'string',
          received: 'number',
        },
      ]);

      const mockResponse = {
        status: jest.fn().mockReturnThis(),
        json: jest.fn(),
      };

      const host = {
        switchToHttp: () => ({
          getResponse: () => mockResponse,
          getRequest: () => ({ url: '/test' }),
        }),
      };

      filter.catch(exception, host as any);

      expect(mockResponse.status).toHaveBeenCalledWith(HttpStatus.BAD_REQUEST);
      expect(mockResponse.json).toHaveBeenCalledWith({
        status: HttpStatus.BAD_REQUEST,
        message: 'Validation error',
        type: ExceptionType.INTERNAL,
      });
    });
  });
});
