import {
  CanActivate,
  ExecutionContext,
  ForbiddenException,
  Inject,
  Injectable,
  UnauthorizedException,
} from '@nestjs/common';
import { Reflector } from '@nestjs/core';
import { Request } from 'express';
import { PrimeLogger } from '../../logger/app.exception.logger';
import { FirebaseService, UserRoleService, UserService } from 'src/framework/application';
import { IS_PUBLIC_KEY } from 'src/framework/infrastructure/decorators/public/public.decorator';
import { Role, User } from 'src/framework/domain';
import { ALLOWED_ROLES, RoleShortNameEnum } from 'src/framework/infrastructure/decorators/allow-profiles/allow-roles.decorator';

@Injectable()
export class PrimeFirebaseAuthGuard implements CanActivate {
  private readonly LOGGER = new PrimeLogger(PrimeFirebaseAuthGuard.name);
  constructor(
    @Inject('FirebaseService') private readonly firebaseService: FirebaseService,
    @Inject('UserService') private readonly userService: UserService,
    @Inject('UserRoleService') private readonly userRoleService: UserRoleService,
    private readonly reflector: Reflector,
  ) {}

  async canActivate(context: ExecutionContext): Promise<boolean> {

    const isPublic = this.reflector.getAllAndOverride<boolean>(
      IS_PUBLIC_KEY, [
      context.getHandler(),
      context.getClass(),
    ]);
    if (isPublic) {
      return true;
    }

    const request = context.switchToHttp().getRequest<Request>();

    const token = this.extractTokenFromHeader(request);

    if (!token) {
      this.LOGGER.error('Token not found in request header');
      throw new UnauthorizedException();
    }

    let user: User | null | undefined;
    let roles: Role[] = [];
    try {
      const decodedToken = await this.firebaseService.verifyIdToken(token);
      request.decodedToken = decodedToken;
      request.accessToken = token;

      user = await this.userService.getUserByExternalId(decodedToken.sub);
      request.user = user ?? undefined;

      if (user) {
        roles = await this.userRoleService.getUserRoles(user.id);
        request.profiles = roles;
      }
    } catch (error) {
      this.LOGGER.error(`Error verifying token: ${error.message}`, error);
      throw new UnauthorizedException(error);
    }

    const allowedUserTypes = this.reflector.get<RoleShortNameEnum[]>(
      ALLOWED_ROLES,
      context.getHandler(),
    );
    if (allowedUserTypes && allowedUserTypes.length > 0) {
      if (!user || !roles || roles.length === 0) {
        throw new UnauthorizedException();
      }

      const allowedProfiles = roles.filter((role) =>
        allowedUserTypes.includes(role.shortName),
      );

      if (allowedProfiles.length === 0) {
        throw new ForbiddenException();
      }
    }
    return true;
  }

  private extractTokenFromHeader(request: Request): string | undefined {
    const [type, token] = request.headers.authorization?.split(' ') ?? [];
    return type === 'Bearer' ? token : undefined;
  }
}
