import { Module, OnModuleDestroy, OnModuleInit } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { drizzle } from 'drizzle-orm/mysql2';
import { createPool } from 'mysql2/promise';
import { MYSQL_CONNECTION } from './drizzle.constants';
import { DrizzleLogger } from './drizzle.logger';
import { DBConfigService } from './drizzle.provider';
import { PrimeLogger } from '../definition/logger/app.exception.logger';


@Module({
  providers: [
    {
      provide: MYSQL_CONNECTION,
      inject: [ConfigService],
      useFactory: async (configService: ConfigService) => {
        const log = new PrimeLogger('DrizzleModule.factory');
        const host = configService.get<string>('DB_HOST');
        const port = configService.get<number>('DB_PORT');
        const database = configService.get<string>('DB_DATABASE');
        const user = configService.get<string>('DB_USERNAME');
        const password = configService.get<string>('DB_PASSWORD');
        const logSql = configService.get<string>('DB_LOG_SQL') === 'true';

        log.debug(`Connecting to ${host}:${port}/${database}`, 'DrizzleModule');

        const pool = createPool({
          host,
          port,
          database,
          user,
          password,
          waitForConnections: true,
          connectionLimit: 20,
          maxIdle: 10, // max idle connections, the default value is the same as `connectionLimit`
          idleTimeout: 30000, // idle connections timeout, in milliseconds, the default value 60000
          queueLimit: 0,
          enableKeepAlive: true,
          keepAliveInitialDelay: 0,
          timezone: '-03:00',
        });

        return {
          connection: pool,
          drizzle: drizzle(pool, {
            logger: new DrizzleLogger(logSql, ':'),
          }),
        };
      },
    },
    DBConfigService,
  ],
  exports: [MYSQL_CONNECTION, DBConfigService],
})
export class DrizzleModule implements OnModuleInit, OnModuleDestroy {
  private readonly LOGGER = new PrimeLogger(DrizzleModule.name);
  constructor(
    private readonly dbConfigService: DBConfigService,
    private readonly configService: ConfigService,
  ) {}
  async onModuleInit() {
    const shouldRunMigrations =
      this.configService
        .get<string>('DB_RUN_MIGRATIONS', 'false')
        .toLowerCase() === 'true';

    const shouldSeed =
      this.configService.get<string>('DB_SEED', 'false').toLowerCase() ===
      'true';

    if (shouldRunMigrations) {
      this.LOGGER.debug('Running migrations', 'DrizzleModule');
      await this.dbConfigService.migrate();
    }
    if (shouldSeed) {
      this.LOGGER.debug('Running seeds', 'DrizzleModule');
      this.dbConfigService.seed().then((results) => {
        results.forEach((result) => {
          this.LOGGER.debug(
            `Seed file ${result.file} executed with ${result.success} success and ${result.error} errors`,
            'DrizzleModule',
          );
        });
      });
    }
  }

  onModuleDestroy() {
    this.dbConfigService.pool.end();
  }
}
