import { Inject, Injectable } from '@nestjs/common';
import { sql } from 'drizzle-orm';
import { MySql2Database } from 'drizzle-orm/mysql2';
import { migrate } from 'drizzle-orm/mysql2/migrator';
import * as fs from 'fs';
import { Pool } from 'mysql2';
import { inspect } from 'node:util';
import * as path from 'path';
import { MYSQL_CONNECTION } from './drizzle.constants';
import * as schema from './migrations/schema';
import { PrimeLogger } from '../definition/logger/app.exception.logger';

@Injectable()
export class DBConfigService {
  private readonly LOGGER = new PrimeLogger(DBConfigService.name);
  constructor(
    @Inject(MYSQL_CONNECTION)
    private readonly config: {
      connection: Pool;
      drizzle: MySql2Database<typeof schema>;
    },
  ) {}

  public get conn() {
    return this.config.drizzle;
  }
  public get pool() {
    return this.config.connection;
  }
  
  public async migrate(): Promise<boolean> {
    return await migrate(this.conn, {
      migrationsFolder: `./src/framework/infrastructure/drizzle/migrations/output`,
    })
      .then(() => {
        this.LOGGER.debug(
          'Migrations executed successfully',
          'DBConfigService',
        );
        return true;
      })
      .catch((error) => {
        this.LOGGER.error(
          `Error executing migrations: ${inspect(error)}`,
          'DBConfigService',
        );
        return false;
      });
  }

  public async seed(): Promise<
    { file: string; error: number; success: number }[]
  > {
    const seedFiles = [
      'ddl_schema.sql',
      'profile.sql',
      'type_master.sql',
      'truncate.sql',
      'continent.sql',
      'country.sql',
      'subdivision.sql',
      'administrative_division.sql',
    ];
    const results: { file: string; error: number; success: number }[] = [];
    const promises: Promise<any>[] = [];

    for (const file of seedFiles) {
      const result = { file, error: 0, success: 0 };
      const filePath = path.join(
        __dirname,
        '..',
        '..',
        '..',
        'infrastructure',
        'drizzle',
        'migrations',
        'seed',
        file,
      );

      const imports = fs.readFileSync(filePath, 'utf8');

      const sqlStatements = imports
        .split(';')
        .map((stmt) => stmt.trim())
        .filter((stmt) => stmt.length > 0);

      for (const statement of sqlStatements) {
        promises.push(
          this.conn
            .execute(sql.raw(statement))
            .then(() => {
              result.success++;
            })
            .catch(() => {
              result.error++;
            }),
        );
      }
      results.push(result);
    }
    await Promise.all(promises);
    return results;
  }
}
export default {DBConfigService};