import { Injectable } from '@nestjs/common';
import { GeoRepository } from 'src/framework/application/repository/geo-repository/geo-repository.interface';
import {
  AdministrativeDivision,
  Country,
} from 'src/framework/domain';
import { DBConfigService } from 'src/framework/infrastructure/drizzle/drizzle.provider';
import {
  administrativeDivisionTable,
  countryTable,
  subdivisionTable,
} from '../../drizzle/migrations/schema';
import { and, asc, eq, like } from 'drizzle-orm';
import { PrimeLogger } from '../../definition';
import { Subdivision } from 'src/framework/domain/entities/subdivision.entity';

@Injectable()
export class GeoRepositoryImpl implements GeoRepository {
  private readonly LOGGER = new PrimeLogger(GeoRepositoryImpl.name);
  constructor(private readonly db: DBConfigService) {}
  getAllAdministrativeDivisions(): Promise<AdministrativeDivision[]> {
    this.LOGGER.log(`getAllAdministrativeDivisions`);
    return this.db.conn
      .select({
        id: administrativeDivisionTable.id,
        name: administrativeDivisionTable.name,
        code: administrativeDivisionTable.code,
        subdivisionId: administrativeDivisionTable.subdivisionId,
      })
      .from(administrativeDivisionTable)
      .where(eq(administrativeDivisionTable.active, true))
      .then((rows) => {
        return rows.map((row) => {
          return new AdministrativeDivision(
            row.id,
            row.code,
            row.name,
            row.subdivisionId,
          );
        });
      });
  }
  getAllSubdivisions(): Promise<Subdivision[]> {
    this.LOGGER.log(`getAllSubdivisions`);
    return this.db.conn
      .select({
        id: subdivisionTable.id,
        name: subdivisionTable.name,
        code: subdivisionTable.code,
      })
      .from(subdivisionTable)
      .where(eq(subdivisionTable.active, true))
      .then((rows) => {
        return rows.map((row) => {
          return new Subdivision(row.id, row.code, row.name);
        });
      });
  }
  async findCountries() {
    return await this.db.conn
      .select({
        id: countryTable.id,
        name: countryTable.name,
        fullName: countryTable.fullName,
        code: countryTable.code,
        iso3: countryTable.iso3,
        number: countryTable.number,
      })
      .from(countryTable)
      .orderBy(asc(countryTable.name))
      .then((rows) => {
        return rows.map((row) => {
          return new Country(
            row.id,
            row.code,
            row.iso3,
            row.number,
            row.name,
            row.fullName,
          );
        });
      });
  }

  async findCountryById(id: number) {
    return await this.db.conn
      .select({
        id: countryTable.id,
        name: countryTable.name,
        fullName: countryTable.fullName,
        code: countryTable.code,
        iso3: countryTable.iso3,
        number: countryTable.number,
      })
      .from(countryTable)
      .where(eq(countryTable.id, id))
      .then((rows) => {
        if (rows.length === 0) {
          return null;
        }
        const row = rows[0];
        return new Country(
          row.id,
          row.code,
          row.iso3,
          row.number,
          row.name,
          row.fullName,
        );
      });
  }

  async findSubdivisionsByCountryCode(countryCode: string) {
    return this.db.conn
      .select({
        id: subdivisionTable.id,
        name: subdivisionTable.name,
        code: subdivisionTable.code,
      })
      .from(subdivisionTable)
      .innerJoin(countryTable, eq(subdivisionTable.countryId, countryTable.id))
      .where(eq(countryTable.code, countryCode))
      .orderBy(asc(subdivisionTable.name))
      .then((rows) => {
        var output = [new Subdivision(7777, 'all', 'Seleccionar todos')];
        for (var i = 0; i < rows.length; i++) {
          output.push(new Subdivision(rows[i].id, rows[i].code, rows[i].name));
        }
        return output;
      });
  }

  async findAdmdivisionsBySubdivisionCode(subdivisionCode: number) {
    return this.db.conn
      .select({
        id: administrativeDivisionTable.id,
        name: administrativeDivisionTable.name,
        code: administrativeDivisionTable.code,
      })
      .from(administrativeDivisionTable)
      .where(eq(administrativeDivisionTable.subdivisionId, subdivisionCode))
      .orderBy(asc(administrativeDivisionTable.name))
      .then((rows) => {
        return rows.map((row) => {
          return new AdministrativeDivision(row.id, row.code, row.name);
        });
      });
  }
  async findAdmdivisionsByCountry(prefixCode: string) {
    this.LOGGER.log(`findAdmdivisionsByCountry - prefixCode: ${prefixCode}`);
    return this.db.conn
      .select({
        id: administrativeDivisionTable.id,
        name: administrativeDivisionTable.name,
        code: administrativeDivisionTable.code,
        subdivisionId: administrativeDivisionTable.subdivisionId,
      })
      .from(administrativeDivisionTable)
      .where(
        and(
          like(administrativeDivisionTable.code, `${prefixCode}%`),
          eq(administrativeDivisionTable.active, true),
        ),
      )
      .then((rows) => {
        if (rows.length === 0) {
          return [];
        }
        return rows.map((row) => {
          return new AdministrativeDivision(
            row.id,
            row.code,
            row.name,
            row.subdivisionId,
          );
        });
      });
  }
}
