import { Injectable } from '@nestjs/common';
import { TypeMasterRepository } from 'src/framework/application/repository/type-master-repository/type-master-repository.interface';
import { InsertTypeMaster, TypeMaster } from 'src/framework/domain';
import { DBConfigService } from 'src/framework/infrastructure/drizzle/drizzle.provider';
import { TxType } from '../../drizzle/drizzle.types';
import * as schema from 'src/framework/infrastructure/drizzle/migrations/schema';
import { eq, and } from 'drizzle-orm';
import { TypeMasterDiscriminatorEnum } from 'src/framework/domain/enum/enum.definition';
import { PrimeLogger } from '../../definition/logger/app.exception.logger';

@Injectable()
export class TypeMasterRepositoryImpl implements TypeMasterRepository {
    private readonly LOGGER = new PrimeLogger(TypeMasterRepositoryImpl.name);
    constructor(private readonly db: DBConfigService) {}
    async findByDiscriminator(discriminator: string, tx?: TxType): Promise<TypeMaster | null > {
        this.LOGGER.log(`findByDiscriminator - discriminator: ${discriminator}`);
        const discriminatorOBJ = TypeMasterDiscriminatorEnum[discriminator as keyof typeof TypeMasterDiscriminatorEnum];
        return await (tx || this.db.conn)
          .select({
            id: schema.typeMasterTable.id,
            name: schema.typeMasterTable.name,
            shortName: schema.typeMasterTable.shortName,
            discriminator: schema.typeMasterTable.discriminator,
            order: schema.typeMasterTable.order,
          })
          .from(schema.typeMasterTable)
          .where( and(eq(schema.typeMasterTable.active, true),
            eq(schema.typeMasterTable.discriminator, discriminatorOBJ)))
          .orderBy(schema.typeMasterTable.order)
          .then((rows) => {
            if (rows.length === 0) {
              return null;
            }
            return new TypeMaster(
              rows[0].id,
              rows[0].name,
              rows[0].shortName,
              rows[0].discriminator,
              rows[0].order,
            );
          });
    }

    async findById(id: number, tx?: TxType): Promise<TypeMaster | null | undefined> {
      this.LOGGER.log(`findById - id: ${id}`);
        return await (tx || this.db.conn)
          .select({
            id: schema.typeMasterTable.id,
            name: schema.typeMasterTable.name,
            shortName: schema.typeMasterTable.shortName,
            discriminator: schema.typeMasterTable.discriminator,
            order: schema.typeMasterTable.order,
          })
          .from(schema.typeMasterTable)
          .where(eq(schema.typeMasterTable.id, id))
          .then((rows) => {
            if (rows.length === 0) {
              return null;
            }
            return new TypeMaster(
              rows[0].id,
              rows[0].name,
              rows[0].shortName,
              rows[0].discriminator,
              rows[0].order,
            );
          });
    }

    async findByShortName(shortName: string, tx?: TxType): Promise<TypeMaster | null> {
      this.LOGGER.log(`findByShortName - shortName: ${shortName}`);
      return await (tx || this.db.conn)
          .select({
            id: schema.typeMasterTable.id,
            name: schema.typeMasterTable.name,
            shortName: schema.typeMasterTable.shortName,
            discriminator: schema.typeMasterTable.discriminator,
            order: schema.typeMasterTable.order,
          })
          .from(schema.typeMasterTable)
          .where(eq(schema.typeMasterTable.shortName, shortName))
          .then((rows) => {
            if (rows.length === 0) {
              return null;
            }
            return new TypeMaster(
              rows[0].id,
              rows[0].name,
              rows[0].shortName,
              rows[0].discriminator,
              rows[0].order,
            );
          });
    }

    async save(typeMaster: InsertTypeMaster, tx?: TxType): Promise<TypeMaster> {
        this.LOGGER.log(`save - typeMaster: ${JSON.stringify(typeMaster)}`);
        const validateTypeMaster = schema.typeMasterTableInsertSchema.parse(typeMaster);
        const typeMasterId = await (tx || this.db.conn)
            .insert(schema.typeMasterTable)
            .values(validateTypeMaster)
            .$returningId()
            .then((rows) => {
                return rows[0].id;
            });
        return await this.findById(typeMasterId, tx).then((typeMaster) => {
            if (typeMaster) {
                return typeMaster;
            }
            throw new Error('TypeMaster not found');
        });
    }

}