import { Inject, Injectable } from '@nestjs/common';
import { UserDeviceRepository } from 'src/framework/application/repository/user-device-repository/user-device-repository.interface';
import { PrimeLogger } from '../../definition/logger/app.exception.logger';
import { DBConfigService } from 'src/framework/infrastructure/drizzle/drizzle.provider';
import { TxType } from '../../drizzle/drizzle.types';
import * as schema from 'src/framework/infrastructure/drizzle/migrations/schema';
import { eq, and, inArray } from 'drizzle-orm';
import { TenderUtil } from 'src/licitaapp/domain/util';
import { UserRoleService } from 'src/framework/application';
import {
  InsertUserDevice,
  UserDevice,
} from 'src/framework/domain/entities/user-device.entity';

@Injectable()
export class UserDeviceRepositoryImpl implements UserDeviceRepository {
  private readonly LOGGER = new PrimeLogger(UserDeviceRepositoryImpl.name);
  constructor(
    private readonly db: DBConfigService,
    @Inject('UserRoleService')
    private readonly userRoleService: UserRoleService,
  ) {}
  async erraseByIds(ids: number[], tx?: TxType): Promise<void> {
    this.LOGGER.log(`Erasing user devices with ids: ${ids.join(', ')}`);
    await (tx || this.db.conn)
      .delete(schema.userDeviceTable)
      .where(inArray(schema.userDeviceTable.id, ids));
  }

  erraseUserDevice(token: string): Promise<void> {
    this.LOGGER.log(`Erasing user device with token: ${token}`);
    return this.db.conn
      .delete(schema.userDeviceTable)
      .where(eq(schema.userDeviceTable.token, token))
      .then(() => {
        this.LOGGER.log(`User device with token: ${token} has been erased`);
      });
  }
  async logicalRemoveById(id: number, tx?: TxType): Promise<void> {
    this.LOGGER.log(`Logically removing user device with id: ${id}`);
    await (tx || this.db.conn)
      .update(schema.userDeviceTable)
      .set({
        active: false,
        updatedAt: TenderUtil.getCurrentSystemDate(),
        deletedAt: TenderUtil.getCurrentSystemDate(),
      })
      .where(eq(schema.userDeviceTable.id, id));
    this.LOGGER.log(`User device with id: ${id} has been logically removed`);
  }
  async findByUserId(userId: number, tx?: TxType): Promise<string[]> {
    this.LOGGER.log(`Finding user device for user id: ${userId}`);
    return await (tx || this.db.conn)
      .select({
        id: schema.userDeviceTable.id,
        userId: schema.userDeviceTable.userId,
        token: schema.userDeviceTable.token,
        information: schema.userDeviceTable.information,
      })
      .from(schema.userDeviceTable)
      .where(
        and(
          eq(schema.userDeviceTable.active, true),
          eq(schema.userDeviceTable.userId, userId),
        ),
      )
      .then((rows) => {
        this.LOGGER.log(
          `Found ${rows.length} userDevices for user id: ${userId}`,
        );
        return rows.map((row) => row.token);
      });
  }
  async findUserWithToken(token: string, tx?: TxType): Promise<UserDevice[]> {
    this.LOGGER.log(`Finding findUserWithToken device by token: ${token}`);
    const results: UserDevice[] = [];
    await (tx || this.db.conn)
      .select({
        id: schema.userDeviceTable.id,
        userId: schema.userDeviceTable.userId,
        token: schema.userDeviceTable.token,
        information: schema.userDeviceTable.information,
      })
      .from(schema.userDeviceTable)
      .where(
        and(
          eq(schema.userDeviceTable.active, true),
          eq(schema.userDeviceTable.token, token),
        ),
      )
      .then((rows) => {
        if (rows.length === 0) {
          return null;
        }
        results.push(
          new UserDevice(
            rows[0].id,
            rows[0].userId,
            rows[0].token,
            rows[0].information,
          ),
        );
      });
    return results;
  }
  async getAdminUserDeviceTokens(): Promise<string[]> {
    this.LOGGER.log(`Finding user device as admin token `);

    const userIds = await this.userRoleService.getAdminUserIds();
    this.LOGGER.warn(`userIds SIZE: ${userIds.length}`);
    return await this.db.conn
      .select({
        token: schema.userDeviceTable.token,
      })
      .from(schema.userDeviceTable)
      .innerJoin(
        schema.userTable,
        eq(schema.userDeviceTable.userId, schema.userTable.id),
      )
      .where(
        and(
          eq(schema.userDeviceTable.active, true),
          inArray(schema.userDeviceTable.userId, userIds),
        ),
      )
      .then((rows) => {
        return rows.map((row) => row.token);
      });
  }

  private getDeviceInfo(deviceInfo: string): {
    model: string;
    idPhone: string;
  } {
    const parts = deviceInfo.split(';');
    return {
      model: parts[0] || '',
      idPhone: parts[parts.length - 1] || '',
    };
  }

  async findById(
    id: number,
    tx?: TxType,
  ): Promise<UserDevice | null | undefined> {
    this.LOGGER.log(`Finding user device by userId: ${id}`);

    return await (tx || this.db.conn)
      .select({
        id: schema.userDeviceTable.id,
        userId: schema.userDeviceTable.userId,
        token: schema.userDeviceTable.token,
        information: schema.userDeviceTable.information,
      })
      .from(schema.userDeviceTable)
      .where(
        and(
          eq(schema.userDeviceTable.active, true),
          eq(schema.userDeviceTable.id, id),
        ),
      )
      .then((rows) => {
        if (rows.length === 0) {
          this.LOGGER.log(`No user device found with id: ${id}`);
          return null;
        }
        this.LOGGER.log(`Found user device with id: ${id}`);
        return new UserDevice(
          rows[0].id,
          rows[0].userId,
          rows[0].token,
          rows[0].information,
        );
      });
  }
  async save(
    userId: number,
    userDevice: InsertUserDevice,
    tx?: TxType,
  ): Promise<UserDevice> {
    this.LOGGER.log(`Saving user device for user id: ${userId}`);

    // Elimina cualquier userDevice existente con el mismo token
    await (tx || this.db.conn)
      .delete(schema.userDeviceTable)
      .where(eq(schema.userDeviceTable.token, userDevice.token));

    // Inserta el nuevo userDevice
    userDevice.userId = userId;
    const validatedUserDevice = schema.userDeviceTableInsertSchema.parse(userDevice);
    const insertedUserDeviceId = await (tx || this.db.conn)
      .insert(schema.userDeviceTable)
      .values(validatedUserDevice)
      .$returningId()
      .then((rows) => {
        this.LOGGER.log(`Inserted User Device with id: ${rows[0].id}`);
        return rows[0].id;
      });

    // Retorna el userDevice insertado
    return await this.findById(insertedUserDeviceId).then((userDevice) => {
      if (userDevice) {
        this.LOGGER.log(
          `User device saved successfully with id: ${userDevice.id}`,
        );
        return userDevice;
      } else {
        this.LOGGER.error('UserDevice not found after save');
        throw new Error('UserDevice not found');
      }
    });
  }
}
