import { Injectable } from '@nestjs/common';
import { UserRepository } from 'src/framework/application';
import { User, InsertUser, UpdateUser, Role } from 'src/framework/domain';
import { DBConfigService } from 'src/framework/infrastructure/drizzle/drizzle.provider';
import { TxType } from 'src/framework/infrastructure/drizzle/drizzle.types';
import * as schema from 'src/framework/infrastructure/drizzle/migrations/schema';
import { and, eq, inArray } from 'drizzle-orm';
import { PrimeLogger } from '../../definition/logger/app.exception.logger';
import { TenderUtil } from 'src/licitaapp/domain/util';

@Injectable()
export class UserRepositoryImpl implements UserRepository {
  private readonly LOGGER = new PrimeLogger(UserRepositoryImpl.name);
  constructor(private readonly db: DBConfigService) {}
  async findAll(): Promise<User[]> {
    this.LOGGER.log('Finding all users');
    return await this.db.conn
      .select({
        id: schema.userTable.id,
        name: schema.userTable.name,
        email: schema.userTable.email,
        lastName: schema.userTable.lastName,
        lastname2: schema.userTable.lastname2,
        cellPhone: schema.userTable.cellPhone,
      })
      .from(schema.userTable)
      .then((rows) => {
        this.LOGGER.log(`Found ${rows.length} users`);
        return rows.map(
          (row) =>
            new User(
              row.id,
              row.name,
              row.email,
              row.lastName,
              row.lastname2,
              row.cellPhone,
            ),
        );
      });
  }

  async updateLastLogin(userId: number): Promise<void> {
    this.LOGGER.log(`Updating last login for user with id: ${userId}`);
    await this.db.conn
      .update(schema.userTable)
      .set({ lastAccess: TenderUtil.getCurrentSystemDate() })
      .where(eq(schema.userTable.id, userId));
  }

  async findEmailsAdmin(userIds:number[]): Promise<string[]> {
    this.LOGGER.log('Finding all users actives admin');
    return await this.db.conn
      .select({
        email: schema.userTable.email,
      })
      .from(schema.userTable)
      .where(and(inArray(schema.userTable.id, userIds), eq(schema.userTable.active, true)))
      .then((rows) => {
        this.LOGGER.log(`Found ${rows.length} users`);
        return rows.map(
          (row) =>
            row.email
        );
      });
  }

  async findByEmail(email: string): Promise<User | null | undefined> {
    this.LOGGER.log(`Finding user by email: ${email}`);
    return await this.db.conn
      .select({
        id: schema.userTable.id,
        name: schema.userTable.name,
        email: schema.userTable.email,
        lastName: schema.userTable.lastName,
        lastname2: schema.userTable.lastname2,
        cellPhone: schema.userTable.cellPhone,
        externalId: schema.userTable.externalId,
        aggrementDate: schema.userTable.agreementAt,
      })
      .from(schema.userTable)
      .where(eq(schema.userTable.email, email))
      .then((rows) => {
        if (rows.length === 0) {
          this.LOGGER.log(`No user found with email: ${email}`);
          return null;
        }
        this.LOGGER.log(`Found user with email: ${email}`);
        return new User(
          rows[0].id,
          rows[0].name,
          rows[0].email,
          rows[0].lastName,
          rows[0].lastname2,
          rows[0].cellPhone,
          rows[0].externalId ? rows[0].externalId : undefined,
          rows[0].aggrementDate ? rows[0].aggrementDate : undefined,
        );
      });
  }

  async findById(id: number, tx?: TxType): Promise<User | null | undefined> {
    this.LOGGER.log(`Finding user by id: ${id}`);
    return await (tx || this.db.conn)
      .select({
        id: schema.userTable.id,
        name: schema.userTable.name,
        email: schema.userTable.email,
        lastName: schema.userTable.lastName,
        lastname2: schema.userTable.lastname2,
        cellPhone: schema.userTable.cellPhone,
        externalId: schema.userTable.externalId,
        aggrementDate: schema.userTable.agreementAt,
      })
      .from(schema.userTable)
      .where(eq(schema.userTable.id, id))
      .then((rows) => {
        if (rows.length === 0) {
          this.LOGGER.log(`No user found with id: ${id}`);
          return null;
        }
        this.LOGGER.log(`Found user with id: ${id}`);
        return new User(
          rows[0].id,
          rows[0].name,
          rows[0].email,
          rows[0].lastName,
          rows[0].lastname2,
          rows[0].cellPhone,
          rows[0].externalId ? rows[0].externalId : undefined,
          rows[0].aggrementDate ? rows[0].aggrementDate : undefined,
        );
      });
  }

  async findByExternalId(externalId: string): Promise<User | null | undefined> {
    this.LOGGER.log(`Finding user by externalId: ${externalId}`);
    return await this.db.conn
      .select({
        id: schema.userTable.id,
        name: schema.userTable.name,
        email: schema.userTable.email,
        lastName: schema.userTable.lastName,
        lastname2: schema.userTable.lastname2,
        cellPhone: schema.userTable.cellPhone,
        externalId: schema.userTable.externalId,
        aggrementDate: schema.userTable.agreementAt,
      })
      .from(schema.userTable)
      .where(eq(schema.userTable.externalId, externalId))
      .then((rows) => {
        if (rows.length === 0) {
          this.LOGGER.log(`No user found with externalId: ${externalId}`);
          return null;
        }
        this.LOGGER.log(`Found user with externalId: ${externalId}`);
        return new User(
          rows[0].id,
          rows[0].name,
          rows[0].email,
          rows[0].lastName,
          rows[0].lastname2,
          rows[0].cellPhone,
          rows[0].externalId ? rows[0].externalId : undefined,
          rows[0].aggrementDate ? rows[0].aggrementDate : undefined,
        );
      });
  }

  async save(user: InsertUser): Promise<User> {
    this.LOGGER.log(`Saving user with email: ${user.email}`);
    const validatedUser = schema.userTableInsertSchema.parse(user);
    const insertedUserId = await this.db.conn
      .insert(schema.userTable)
      .values(validatedUser)
      .$returningId()
      .then((rows) => {
        this.LOGGER.log(`Inserted user with id: ${rows[0].id}`);
        return rows[0].id;
      });

    return await this.findById(insertedUserId).then((user) => {
      if (user) {
        this.LOGGER.log(`User saved successfully with id: ${user.id}`);
        return user;
      }
      this.LOGGER.error('User not found after save');
      throw new Error('User not found');
    });
  }

  async update(id: number, user: UpdateUser): Promise<User> {
    this.LOGGER.log(`Updating user with id: ${id}`);
    const validatedUser = schema.userTableUpdateSchema.parse(user);
    await this.db.conn
      .update(schema.userTable)
      .set({ ...validatedUser, updatedAt: TenderUtil.getCurrentSystemDate() })
      .where(eq(schema.userTable.id, id));

    return await this.findById(id).then((user) => {
      if (user) {
        this.LOGGER.log(`User updated successfully with id: ${user.id}`);
        return user;
      } else {
        this.LOGGER.error('User not found after update');
        throw new Error('User not found');
      }
    });
  }

  async findUserProfiles(userId: number): Promise<Role[]> {
    this.LOGGER.log(`Finding profiles for user id: ${userId}`);
    return await this.db.conn
      .select({
        id: schema.roleTable.id,
        name: schema.roleTable.name,
        shortName: schema.roleTable.shortName,
      })
      .from(schema.roleTable)
      .innerJoin(
        schema.userRoleTable,
        eq(schema.roleTable.id, schema.userRoleTable.roleId),
      )
      .where(eq(schema.userRoleTable.userId, userId))
      .then((rows) => {
        this.LOGGER.log(`Found ${rows.length} profiles for user id: ${userId}`);
        return rows.map((row) => new Role(row.id, row.name, row.shortName));
      });
  }

  async updateAggrement(user : User): Promise<string> {
    this.LOGGER.log(`Updating user agreement with id: ${user.id}`);
    await this.db.conn
      .update(schema.userTable)
      .set({ updatedAt: TenderUtil.getCurrentSystemDate(), agreementAt: TenderUtil.getCurrentSystemDate(),
        cellPhone: user.cellPhone
       })
      .where(eq(schema.userTable.id, user.id));
    return 'ok';
  }

  async unsubscribe(userId: number): Promise<User> {
    this.LOGGER.log(`Unsubscribing user with id: ${userId}`);
    await this.db.conn
      .update(schema.userTable)
      .set({ deletedAt: TenderUtil.getCurrentSystemDate(), active: false })
      .where(eq(schema.userTable.id, userId));
    return await this.findById(userId).then((user) => {
      if (user) {
        this.LOGGER.log(`User unsubscribed successfully with id: ${user.id}`);
        return user;
      } else {
        this.LOGGER.error('User not found after unsubscribe');
        throw new Error('User not found');
      }
    });
  }

  async erraseAccount(userId: number): Promise<boolean> {
    this.LOGGER.log(`erraseAccount id: ${userId}`);

    await this.db.conn.transaction(async (tx) => {
      await tx
      .delete(schema.userDeviceTable)
      .where(eq(schema.userDeviceTable.userId, userId));
      this.LOGGER.log(`Deleted userDeviceTable records for userId: ${userId}`);

      await tx
      .delete(schema.userCompanyTenderTable)
      .where(eq(schema.userCompanyTenderTable.userId, userId));
      this.LOGGER.log(`Deleted userCompanyTenderTable records for userId: ${userId}`);

      await tx
      .delete(schema.userCompanyTable)
      .where(eq(schema.userCompanyTable.userId, userId));
      this.LOGGER.log(`Deleted userCompanyTable records for userId: ${userId}`);

      await tx
      .delete(schema.notificationRecordTable)
      .where(eq(schema.notificationRecordTable.userId, userId));
      this.LOGGER.log(`Deleted notificationRecordTable records for userId: ${userId}`);

      await tx
      .delete(schema.userHistoryTenderTable)
      .where(eq(schema.userHistoryTenderTable.userId, userId));
      this.LOGGER.log(`Deleted userHistoryTenderTable records for userId: ${userId}`);

      await tx
      .delete(schema.userRoleTable)
      .where(eq(schema.userRoleTable.userId, userId));
      this.LOGGER.log(`Deleted userRoleTable records for userId: ${userId}`);

      await tx
      .delete(schema.userRequestTable)
      .where(eq(schema.userRequestTable.userId, userId));
      this.LOGGER.log(`Deleted userRequestTable records for userId: ${userId}`);

      await tx
      .delete(schema.userTable)
      .where(eq(schema.userTable.id, userId));
      this.LOGGER.log(`Deleted userTable record for userId: ${userId}`);
    });

    return true;
  }
}