import { Inject, Injectable } from '@nestjs/common';
import { GeoRepository } from 'src/framework/application/repository/geo-repository/geo-repository.interface';
import { GeoService } from 'src/framework/application/service/geo-service/geo-service.interface';
import { PrimeLogger } from '../../definition';
import { AdministrativeDivision } from 'src/framework/domain';
import { Subdivision } from 'src/framework/domain/entities/subdivision.entity';

@Injectable()
export class GeoServiceImpl implements GeoService {
  private readonly LOGGER = new PrimeLogger(GeoServiceImpl.name);
  constructor(
    @Inject('GeoRepository') private readonly repository: GeoRepository,
  ) {}
  async searchSubdivisionsByName(namesToSearch: string[]): Promise<{ subdivision: Subdivision; administrativeDivision: AdministrativeDivision; } | undefined> {
    this.LOGGER.log(`searchSubdivisionsByName`);

    const normalizeText = (text: string): string =>
      text
        .toLowerCase()
        .normalize('NFD')
        .replace(/[\u0300-\u036f]/g, '')
        .replace(/[^a-z0-9 ]/g, '')
        .replace('de', '')
        .replace(' y ', '');
    
    const allSubdivisions = await this.repository.getAllSubdivisions();
    const getAllAdministrativeDivisions = await this.repository.getAllAdministrativeDivisions();
    
    const normalizedNames = namesToSearch.map(normalizeText);

    //for (const nameToFind of namesToSearch) {
      //this.LOGGER.log(`  - nameToFind: ${nameToFind}`);

      for (const subdivision of allSubdivisions) {
        const normalizedSubdivisionName = normalizeText(subdivision.name);

        const matchWithSubdivision = normalizedNames.some(
          (name) =>
            normalizedSubdivisionName.includes(name) ||
            name.includes(normalizedSubdivisionName),
        );

        if (matchWithSubdivision) {
          const administrativeDivision = getAllAdministrativeDivisions.find(
            (admDiv) => admDiv.id === subdivision.id,
          );
          if (administrativeDivision) {
            return { subdivision, administrativeDivision };
          }
        }
      }


    return undefined;
  }
  async findCountries() {
    this.LOGGER.log('Finding all countries');
    return await this.repository.findCountries();
  }

  async findCountryById(id: number) {
    this.LOGGER.log(`Finding country by id: ${id}`);
    return await this.repository.findCountryById(id);
  }

  async findSubdivisionsByCountryCode(countryCode: string) {
    this.LOGGER.log(`Finding subdivisions by country code: ${countryCode}`);
    return await this.repository.findSubdivisionsByCountryCode(countryCode);
  }

  async findAdmdivisionsBySubdivisionCode(subdivisionCode: number) {
    this.LOGGER.log(
      `Finding administrative divisions by subdivision code: ${subdivisionCode}`,
    );
    return await this.repository.findAdmdivisionsBySubdivisionCode(
      subdivisionCode,
    );
  }
  async findAdmdivisionsByCountry(prefix: string) {
    this.LOGGER.log(
      `Finding administrative divisions by country prefix: ${prefix}`,
    );
    return await this.repository.findAdmdivisionsByCountry(prefix);
  }
}
