import { Inject, Injectable } from '@nestjs/common';
import {
  EmailService,
  FirebaseService,
  InsertNotificationRecord,
  InsertUserRequest,
  PrimeLogger,
  ProgramUserRequestTO,
  RequestTenderTO,
  TypeMasterService,
  UserRequest,
  UserRequestService,
  UserRoleService,
  UserService,
} from 'src/framework';
import { UserRequestRepository } from 'src/framework/application/repository/user-request-repository/user-request-repository.interface';
import { NotificationRecordService } from 'src/framework/application/service/notification-record-service/notification-record-service.interface';
import { CompanyService } from 'src/licitaapp';
import { TenderTO } from 'src/licitaapp/domain';

@Injectable()
export class UserRequestServiceImpl implements UserRequestService {
  private readonly LOGGER = new PrimeLogger(UserRequestServiceImpl.name);
  constructor(
    @Inject('UserRequestRepository')
    private readonly userRequestRepository: UserRequestRepository,
    @Inject('TypeMasterService')
    private readonly typeMasterService: TypeMasterService,
    @Inject('UserService') private readonly userService: UserService,
    @Inject('CompanyService') private readonly companyService: CompanyService,
    @Inject('EmailService') private readonly emailService: EmailService,
    @Inject('NotificationRecordService')
    private readonly notificationRecordService: NotificationRecordService,
    @Inject('UserRoleService')
    private readonly userRoleService: UserRoleService,
    @Inject('FirebaseService')
    private readonly firebaseService: FirebaseService,
  ) {}
  countActiveUserRequests(): Promise<number> {
    this.LOGGER.log('Counting active user requests');
    return this.userRequestRepository.countActiveUserRequests();
  }
  erraseUserRequests(tenderIds: TenderTO[]): Promise<void> {
    this.LOGGER.log(`Erasing user requests for tender IDs: ${tenderIds.join(', ')}`);
    return this.userRequestRepository.erraseUserRequests(tenderIds);
  }

  async createProgramUserRequest(
    userId: number,
    dataUserRequest: ProgramUserRequestTO,
  ): Promise<String> {
    this.LOGGER.log(`Creating program user request for user: ${userId}`);

    const [company, user] = await Promise.all([
      this.companyService.findById(dataUserRequest.companyId),
      this.userService.getUserById(userId),
    ]);

    const notificationsArray: InsertNotificationRecord[] = [
      {
        userId: userId,
        title: `Asistencia Programa de Integridad`,
        defaultMessage: `Pronto serás contactado por algún analista.`,
        active: true,
        isAdmin: false,
      },
    ];

    await this.userRequestRepository.save(
      {
        companyId: dataUserRequest.companyId,
        title: dataUserRequest.title,
        description: dataUserRequest.description,
        stateRequestTypeId: 5,
        tenderId: null,
      },
      userId,
    );
    this.LOGGER.log(`Sending email to user: ${user!.id}`);
    this.emailService.sendUserRequest(user!.email, user!.name);

    const adminIds = await this.userRoleService.getAdminUserIds();
    adminIds.forEach((adminId) => {
      notificationsArray.push({
        userId: adminId,
        title: `Programa de Integridad`,
        defaultMessage: `El cliente ${user?.name || ''} ${user?.lastName || ''} razón social ${company?.socialReason || ''} ha solicitado asistencia al Programa de Integridad.`,
        active: true,
        isAdmin: true,
      });
    });

    notificationsArray.length > 0 &&
      (await this.notificationRecordService.saveAll(notificationsArray));

    return 'true';
  }
  async createUserRequest(
    userId: number,
    dataUserRequest: InsertUserRequest,
  ): Promise<UserRequest | null> {
    this.LOGGER.log(`Creating user request for user: ${userId}`);
    const userRequestId = await this.userRequestRepository.existUserRequest(
      userId,
      dataUserRequest.tenderId!,
      dataUserRequest.companyId,
    );
    let companyName = '';
    let fullName = '';
    if (dataUserRequest.companyId) {
      const company = await this.companyService.findById(
        dataUserRequest.companyId,
      );
      companyName = company?.socialReason || '';
      const user = await this.userService.getUserById(userId);
      if (user) {
        fullName =
          user &&
          `${user.name} ${user.lastName != null ? user.lastName : ''} ${user.lastname2 != null ? user.lastname2 : ''}`.trim();
      }
    }
    if (userRequestId > 0) {
      await this.notificationRecordService.saveAll([
        {
          userId: userId,
          title: `Solicitud ${companyName}`,
          defaultMessage: `El cliente ${fullName} ha vuelvo a solicitar apoyo de una licitación.`,
          active: true,
          tenderId: dataUserRequest.tenderId,
          isAdmin: true,
        },
      ]);

      this.firebaseService.groupNotifications(userId);

      this.LOGGER.log(
        `Updating existing user request with id: ${userRequestId}`,
      );
      return this.userRequestRepository.update(userRequestId, dataUserRequest);
    }
    this.LOGGER.log(`Creating new user request`);

    await this.notificationRecordService.saveAll([
      {
        userId: userId,
        title: `Solicitud ${companyName}`,
        defaultMessage: `El cliente ${fullName} ha levantado una solicitud de seguimiento.`,
        active: true,
        tenderId: dataUserRequest.tenderId,
        isAdmin: true,
      },
    ]);
    this.firebaseService.groupNotifications(userId);

    return this.userRequestRepository.save(dataUserRequest, userId);
  }

  async getPagination(
    page: number,
    pageSize: number,
    filter: string,
    shortStateUserRequest: string,
  ): Promise<RequestTenderTO[]> {
    return this.userRequestRepository.getPagination(
      page,
      pageSize,
      shortStateUserRequest,
      filter,
    );
  }

  async deleteLogicalRemove(userRequestId: number): Promise<void> {
    this.LOGGER.log(`Deleting user request: ${userRequestId}`);
    await this.userRequestRepository.logicalRemove(userRequestId);
  }

  async updateUserStateOrLogicalRemove(
    userRequestId: number,
    stateRequestType: string,
  ): Promise<UserRequest | null> {
    this.LOGGER.log(`Updating user request: ${userRequestId}`);

    const typeUserStatus =
      await this.typeMasterService.findByShortName(stateRequestType);

    if (!typeUserStatus) {
      this.LOGGER.error(`State type not found: ${stateRequestType}`);
      throw new Error(`State type not found: ${stateRequestType}`);
    }
    const active =
      stateRequestType == 'state.request.type.deleted' ? true : false;
    return this.userRequestRepository.updateState(
      userRequestId,
      typeUserStatus?.id,
      active,
    );
  }
}
