import { Inject, Injectable } from '@nestjs/common';
import { FwkCacheService, RoleService, UserRoleRepository, UserRoleService } from 'src/framework/application';
import { Role, UserPage, UserRoleTO } from 'src/framework/domain';
import { RoleShortNameEnum } from '../../decorators/allow-profiles/allow-roles.decorator';
import { PrimeLogger } from '../../definition';
import { CacheManagerEnum, FwkCacheServiceImpl } from '../fwk-cache-service-impl/fwk-cache-service-impl';

@Injectable()
export class UserRoleImplService implements UserRoleService {
  private readonly LOGGER = new PrimeLogger(UserRoleImplService.name);
  constructor(
    @Inject('UserRoleRepository') private readonly userRoleRepository: UserRoleRepository,
    @Inject('RoleService') private readonly roleService: RoleService,
    @Inject('FwkCacheService') private readonly fwkCacheService: FwkCacheService,
    ) {}
  async getUserRoles(userId: number): Promise<Role[]> {
    this.LOGGER.log(`Fetching roles for userId: ${userId}`);
    return await this.fwkCacheService.getOrSet<Role[]>(
      FwkCacheServiceImpl.createKey(CacheManagerEnum.USER_ROLES, [userId]),
      async () => {
          return this.userRoleRepository.getUserRoles(userId)
      },
    );
  }
  async getAdminUserIds(): Promise<number[]> {
    this.LOGGER.log('Fetching admin user IDs');
    return await this.userRoleRepository.getAdminUserIds();
  }
  async getAllPaginated(page: number, pageSize: number, email ?: string, profile?: RoleShortNameEnum,): Promise<UserPage[]> {
    this.LOGGER.log(`Fetching paginated user roles for page: ${page}, pageSize: ${pageSize}, profile: ${profile} email: ${email}`);
    const profileObj = await this.roleService.getRoleByKey(profile!);
    if (!profileObj) {
      throw new Error(`Profile with short name ${profile} not found`);
    }
    return await this.userRoleRepository.getAllPaginated(page, pageSize, profileObj.id, email);
  }
  async updateProfiles(params: UserRoleTO, profileId: number): Promise<void> {
    this.LOGGER.log(`Updating user roles for userId: ${params.userId}, profileId: ${profileId}, isAdded: ${params.isAdded}`);
    await this.userRoleRepository.updateRoles(params, profileId);
    await this.fwkCacheService.del();

  }
  async getRolesByUserId(userId: number): Promise<string[]> {
    this.LOGGER.log(`Fetching roles for userId: ${userId}`);
    return await this.userRoleRepository.getRolesByUserId(userId);
  }
  async createUserRole(userId: number, roleId: number): Promise<boolean> {
    this.LOGGER.log(`Creating user role with userId: ${userId}, roleId: ${roleId}`);
    const result = await this.userRoleRepository.save(userId, roleId);
    await this.fwkCacheService.del();
    return result;
  }
}
