import { Inject, Injectable } from '@nestjs/common';
import { InsertUser, User, UpdateUser, Role, RecordTO, UserRoleTO } from 'src/framework/domain';
import { TxType } from 'src/framework/infrastructure/drizzle/drizzle.types';
import { EmailService, FirebaseService, RoleService, UserRepository, UserRoleService, UserService } from 'src/framework/application';
import { PrimeLogger } from '../../definition/logger/app.exception.logger';

@Injectable()
export class UserServiceImpl implements UserService {
  private readonly LOGGER = new PrimeLogger(UserServiceImpl.name);
  constructor(
    @Inject('UserRepository') private readonly userRepository: UserRepository,
    @Inject('FirebaseService') private readonly firebaseService: FirebaseService,
    @Inject('UserRoleService') private readonly userRoleService: UserRoleService,
    @Inject('RoleService') private readonly roleService: RoleService,
    @Inject('EmailService') private readonly emailService: EmailService,
  ) {}
  updateLastLogin(id: number): Promise<void> {
    this.LOGGER.log(`Updating last login for user id: ${id}`);
    return this.userRepository.updateLastLogin(id);
  }
  async updateProfiles(params: UserRoleTO): Promise<string> {
    this.LOGGER.log(`Updating profiles for user id: ${params.userId}`);
    const profile = await this.roleService.getRoleByKey(
      params.profile,
    );
    await this.userRoleService.updateProfiles(params, profile.id);
    return 'true';
  }
  async findEmailsAdmin(): Promise<string[]> {
    const userIds = await this.userRoleService.getAdminUserIds();
    return await this.userRepository.findEmailsAdmin(userIds);
  }
  async createUser(user: InsertUser): Promise<User> {
    this.LOGGER.log(`Creating user with email: ${user.email}`);
    return await this.userRepository.save(user);
  }
  async createOrUpdate(externalId: string, user: InsertUser): Promise<User> {
    this.LOGGER.log(`Creating or updating user with externalId: ${externalId}`);
    const existingUser = await this.userRepository.findByExternalId(externalId);
    if (existingUser) {
      this.LOGGER.log(`Updating existing user with id: ${existingUser.id}`);
      return await this.updateUser(existingUser.id, user);
    }
    const newUser = await this.createUser(user);
    this.LOGGER.log(`Created new user with id: ${newUser.id}`);

    this.LOGGER.log(`Sending welcome email to: ${newUser.email}`);
      this.emailService.sendEmail({
        to: [newUser.email],
        subject: 'LICITAAPP Bienvenido',
        text: 'Registro exitoso',
      });
    return newUser;
  }
  async getUserById(id: number, tx?: TxType): Promise<User | null | undefined> {
    this.LOGGER.log(`Getting user by id: ${id}`);
    return await this.userRepository.findById(id, tx);
  }
  async getUserByExternalId(externalId: string): Promise<User | null | undefined> {
    this.LOGGER.log(`Getting user by externalId: ${externalId}`);
    return await this.userRepository.findByExternalId(externalId);
  }
  async getUserByEmail(email: string): Promise<User | null | undefined> {
    this.LOGGER.log(`Getting user by email: ${email}`);
    return await this.userRepository.findByEmail(email);
  }
  async updateUser(id: number, user: UpdateUser): Promise<User> {
    this.LOGGER.log(`Updating user with id: ${id}`);
    return await this.userRepository.update(id, user);
  }
  async getAllUsers(): Promise<User[]> {
    this.LOGGER.log('Getting all users');
    return await this.userRepository.findAll();
  }
  async updateAggrement(user: User): Promise<string> {
    this.LOGGER.log(`Updating user agreement for user id: ${user.id}`);
    return await this.userRepository.updateAggrement(user);
  }
  async unsubscribe(user: User): Promise<void> {
    this.LOGGER.log(`Unsubscribing user with id: ${user.id} externalid: ${user.externalId}`);
    if(user.externalId) await this.firebaseService.deleteUserFromFirebase(user.externalId);
    await this.userRepository.unsubscribe(user.id);
  }
  async registerUserByProfile(recordTO: RecordTO): Promise<boolean> {
    this.LOGGER.log(`Registering user by profile for email: ${recordTO.user.email}`);
    const user = await this.createUser(recordTO.user);
    const profile = await this.roleService.getRoleByKey(
      recordTO.role.shortName,
    );
    const isValidRegister = await this.userRoleService.createUserRole(user.id, profile.id);
    if (isValidRegister) {
      this.LOGGER.log(`Sending welcome email to: ${recordTO.user.email}`);
      this.emailService.sendEmail({
        to: [recordTO.user.email],
        subject: 'LICITAAPP Bienvenido',
        text: 'Registro exitoso',
      });
    }
    return isValidRegister;
  }
  async errase(user: User): Promise<void> {
    this.LOGGER.log(`errase user with id: ${user.id} externalid: ${user.externalId}`);
    if(user.externalId) await this.firebaseService.deleteUserFromFirebase(user.externalId);
    this.LOGGER.log(`Deleting user with id: ${user.id} from database`);
    await this.userRepository.erraseAccount(user.id);
  }




}
