import { HttpService } from '@nestjs/axios';
import { Injectable } from '@nestjs/common';
import { firstValueFrom, map } from 'rxjs';
import { convertDates } from './mp.util';
import * as moment from 'moment-timezone';
import {

  AgileTenderDetailResponse,
  AgileTenderPageResponse,
  DetailAgileTenderTO,
  LicitacionesQueryParams,
  LicitacionesQueryParamsSchema,
  LicitacionesResponse,
  ListaEmpresasResponse,
  OrdenesDeCompraQueryParams,
  OrdenesDeCompraQueryParamsSchema,
  OrdenesDeCompraResponse,
} from './types';
import { PrimeLogger } from 'src/framework';
import { AxiosResponse } from 'axios';
import { TenderFetcherServiceImpl } from 'src/licitaapp/infrastructure/service/tender-fetcher-service-impl/tender-fetcher-service-impl';

@Injectable()
export class MercadoPublicoRepository {
  private readonly LOGGER = new PrimeLogger(MercadoPublicoRepository.name);
  constructor(
    private readonly httpService: HttpService
  ) {}

  async licitaciones(
    args: LicitacionesQueryParams,
  ): Promise<LicitacionesResponse | undefined> {
    //LicitacionesQueryParamsSchema.parse(args);

    const urlGet = args.fecha?`https://api.mercadopublico.cl/servicios/v1/publico/licitaciones.json?fecha=${args.fecha}&ticket=5DB9D72A-EA99-434D-B0F2-891BA3607F04`
    :`https://api.mercadopublico.cl/servicios/v1/publico/licitaciones.json?ticket=5DB9D72A-EA99-434D-B0F2-891BA3607F04&codigo=${args.codigo}`;
    this.LOGGER.warn(`USANDO URL: ${urlGet}`);
    const response = await firstValueFrom(
      this.httpService
        .get(urlGet)
        .pipe(
          map((response) =>
            LicitacionesResponse.fromJson(convertDates(response.data)),
          ),
        ),
    ).catch((error) => {
      if(error.response){
        this.LOGGER.error(
          `Error response: ${JSON.stringify(error.response.data)}`,
          'MercadoPublicoRepository',
          error
        );
      }
      this.LOGGER.error(
        `Error retrieving licitaciones. ${error.message}. Params: ${JSON.stringify(args)}`,
        'MercadoPublicoRepository',
        error
      );
      //throw error;
      return undefined;
    });
    return response;
  }

  async licitacionesListadoDiarioGeneral(date: string): Promise<LicitacionesResponse | undefined> {
    const delay = (ms: number) =>
      new Promise((resolve) => setTimeout(resolve, ms));
    for (let attempt = 1; attempt <= 3; attempt++) {
      try {
        const response: AxiosResponse = await firstValueFrom(
          this.httpService.get(`https://api.mercadopublico.cl/servicios/v1/publico/licitaciones.json?fecha=${date}&ticket=5DB9D72A-EA99-434D-B0F2-891BA3607F04`),
        );
        return response.data as LicitacionesResponse;
      } catch (error) {
        if (attempt === Number(3)) {
          this.LOGGER.debug(`Failed to fetch tender details for date: ${date}`);
          //throw error;
          this.LOGGER.error(
            `Error retrieving licitaciones for date ${date}. ${error.message}.`,
            'MercadoPublicoRepository',
            error
          );
        }
        this.LOGGER.debug(
          `Attempt ${attempt} failed. Retrying in ${attempt * 1000}ms...`
        );
        await delay(attempt * 1000); // Exponential backoff
      }
    }
  }

  async licitacionesDetalleGeneral(codigo: string): Promise<LicitacionesResponse | undefined> {
    const delay = (ms: number) =>
      new Promise((resolve) => setTimeout(resolve, ms));
    for (let attempt = 1; attempt <= 3; attempt++) {
      try {
        const response: AxiosResponse = await firstValueFrom(
          this.httpService.get(`https://api.mercadopublico.cl/servicios/v1/publico/licitaciones.json?ticket=5DB9D72A-EA99-434D-B0F2-891BA3607F04&codigo=${codigo}`),
        );
        return response.data as LicitacionesResponse;
      } catch (error) {
        if (attempt === Number(3)) {
          this.LOGGER.debug(`Failed to fetch tender details for codigo: ${codigo}`);
          //throw error;
          this.LOGGER.error(
            `Error retrieving licitaciones for date ${error.message}.`,
            'MercadoPublicoRepository',
            error
          );
        }
        this.LOGGER.debug(
          `Attempt ${attempt} failed. Retrying in ${attempt * 1000}ms...`
        );
        await delay(attempt * 1000); // Exponential backoff
      }
    }
  }

  async licitacionesAgiles(args: { initDate: string; endDate: string; page: number; }): Promise<AgileTenderPageResponse | undefined> {
    const { initDate, endDate, page } = args;
    try {
      const response: AxiosResponse = await firstValueFrom(
        this.httpService
          .get(`https://api.buscador.mercadopublico.cl/compra-agil?date_from=${initDate}&date_to=${endDate}&order_by=recent&page_number=${page}&status=2`, {
          headers: {
            'x-api-key': 'e93089e4-437c-4723-b343-4fa20045e3bc',
          },
        }),
      );
      return response.data as AgileTenderPageResponse;
    } catch (error) {
      this.LOGGER.error(
        `Error retrieving licitaciones agiles. ${error.message}. Params: ${JSON.stringify(args)}`,
        'MercadoPublicoRepository',
        error
      );
      //throw error;
      this.LOGGER.error(
            `Error retrieving licitaciones for ${error.message}.`,
            'MercadoPublicoRepository',
            error
          );
          return undefined;
    }
  }

  async licitacionAgilDetail(codeAgileTender: string): Promise<DetailAgileTenderTO | undefined> {
    try {
      const response: AxiosResponse = await firstValueFrom(
        this.httpService
          .get(`https://api.buscador.mercadopublico.cl/compra-agil?action=ficha&code=${encodeURIComponent(codeAgileTender)}`, {
          headers: {
            'x-api-key': 'e93089e4-437c-4723-b343-4fa20045e3bc',
          },
        }),
      );
      const data = response.data as AgileTenderDetailResponse;
      return data.payload;
    } catch (error) {
      this.LOGGER.error(
        `Error retrieving licitaciones agiles. ${error.message}. Params: ${codeAgileTender}`,
        'MercadoPublicoRepository',
        error
      );
      throw error;
    }
  }

  async matchAgileKeyword(codeMercadoPublico: number){
    try {
      const now = moment().tz(TenderFetcherServiceImpl.TIMEZONE);
      const tomorrow = now.add(1, 'day').format('YYYY-MM-DD');
      const lastMonth = moment().tz(TenderFetcherServiceImpl.TIMEZONE).subtract(30, 'days').format('YYYY-MM-DD');
      const response: AxiosResponse = await firstValueFrom(
        this.httpService
          .get(`https://api.buscador.mercadopublico.cl/compra-agil?date_from=${lastMonth}&date_to=${tomorrow}&order_by=recent&category=${codeMercadoPublico}&status=2`, {
          headers: {
            'x-api-key': 'e93089e4-437c-4723-b343-4fa20045e3bc',
          },
        }),
      );
      return response.data as AgileTenderPageResponse;
    } catch (error) {
      this.LOGGER.error(
        `Error retrieving licitaciones agiles. ${error.message}. Params: ${JSON.stringify(codeMercadoPublico)}`,
        'MercadoPublicoRepository',
        error
      );
      throw error;
    }
  }

  async ordenesDeCompra(
    args: OrdenesDeCompraQueryParams,
  ): Promise<OrdenesDeCompraResponse | undefined> {
    OrdenesDeCompraQueryParamsSchema.parse(args);

    const response = await firstValueFrom(
      this.httpService
        .get<OrdenesDeCompraResponse>('/ordenesdecompra.json', {
          params: args,
        })
        .pipe(
          map((response) =>
            OrdenesDeCompraResponse.fromJson(convertDates(response.data)),
          ),
        ),
    ).catch((error) => {
      this.LOGGER.error(
        `Error retrieving ordenes de compra. ${error.message}. Params: ${JSON.stringify(args)}`,
        'MercadoPublicoRepository',
      );
      throw error;
    });
    return response;
  }

  async buscarProveedor(
    rut: string,
  ): Promise<ListaEmpresasResponse | undefined> {
    const response = await firstValueFrom(
      this.httpService
        .get<ListaEmpresasResponse>('/empresas/BuscarProveedor', {
          params: {
            rutempresaproveedor: rut,
          },
        })
        .pipe(map((response) => response.data)),
    ).catch((error) => {
      this.LOGGER.error(
        `Error: ${JSON.stringify(error)}`,
        'MercadoPublicoRepository',
      );
      return undefined;
    });
    return response;
  }

  async buscarComprador(): Promise<ListaEmpresasResponse | undefined> {
    const response = await firstValueFrom(
      this.httpService
        .get<ListaEmpresasResponse>('/empresas/BuscarComprador')
        .pipe(map((response) => response.data)),
    ).catch((error) => {
      this.LOGGER.error(
        `Error: ${JSON.stringify(error)}`,
        'MercadoPublicoRepository',
      );
      return undefined;
    });
    return response;
  }
}
