export class TenderUtil {
  static readonly TIMEZONE = 'America/Santiago';
  static convertToMillions(amount: number): string {
    if (amount >= 1000000) {
      return (
        Math.floor(amount / 1000000)
          .toString()
          .replace(/\B(?=(\d{3})+(?!\d))/g, '.') + 'M'
      );
    }
    return amount.toString().replace(/\B(?=(\d{3})+(?!\d))/g, '.');
  }

  static calculateTimeFromDateString(endDate: string | undefined): string {
    if (!endDate) {
      return '';
    }
    const currentTime = TenderUtil.getCurrentSystemDate();
    const compareDate = new Date(endDate);
    compareDate.toLocaleString('es-CL', { timeZone: 'America/Santiago' });
    const diff = compareDate.getTime() - currentTime.getTime();
    const diffInMinutes = Math.floor(diff / 60000);
    if (diffInMinutes < 0) {
      return 'Finalizado';
    }
    if (diffInMinutes < 60) {
      return diffInMinutes + (diffInMinutes == 1 ? ' minuto' : ' minutos');
    }
    const diffInHours = Math.floor(diff / (1000 * 60 * 60));
    if (diffInHours < 24) {
      return diffInHours + (diffInHours == 1 ? ' hora' : ' horas');
    }
    const diffInDays = Math.floor(diff / (1000 * 60 * 60 * 24));
    return diffInDays + (diffInDays == 1 ? ' día' : ' días');
  }

  static calculateTime(createDate: Date, updateDate?: Date | null): string {
    const currentTime = TenderUtil.getCurrentSystemDate();
    const compareDate = new Date(updateDate ? updateDate : createDate);
    compareDate.toLocaleString('es-CL', { timeZone: 'America/Santiago' });
    const diff = currentTime.getTime() - compareDate.getTime();
    const diffInMinutes = Math.floor(diff / 60000);
    if (diffInMinutes < 60) {
      return diffInMinutes + (diffInMinutes == 1 ? ' minuto' : ' minutos');
    }
    const diffInHours = Math.floor(diff / (1000 * 60 * 60));
    if (diffInHours < 24) {
      return diffInHours + (diffInHours == 1 ? ' hora' : ' horas');
    }
    const diffInDays = Math.floor(diff / (1000 * 60 * 60 * 24));
    return diffInDays + (diffInDays == 1 ? ' día' : ' días');
  }

  static getMonthStartAndEnd(
    date: Date,
    offsetMonths: number = 0,
  ): { start: Date; end: Date } {
    const year = date.getFullYear();
    const month = date.getMonth() + offsetMonths;

    const start = new Date(year, month, 1, 0, 0, 0);
    start.toLocaleString('es-CL', { timeZone: 'America/Santiago' });
    const end = new Date(year, month + 1, 0, 23, 59, 59);
    end.toLocaleString('es-CL', { timeZone: 'America/Santiago' });
    return { start, end };
  }

  static subtractDays(date: Date, days: number): Date {
    const result = new Date(date);
    result.setDate(result.getDate() - days);
    return result;
  }

  static getDayStartAndEnd(date: Date): { start: Date; end: Date } {
    const start = new Date(date);
    start.toLocaleString('es-CL', { timeZone: 'America/Santiago' });
    start.setHours(0, 0, 0, 0);

    const end = new Date(date);
    end.toLocaleString('es-CL', { timeZone: 'America/Santiago' });
    end.setHours(23, 59, 59, 999);

    return { start, end };
  }

  /**
   * return the current date with time set to 00:00:00
   * @returns
   */
  static getSystemDateDDMMYYYY(): Date {
    const now = new Date();
    return new Date(
      now.getFullYear(),
      now.getMonth(),
      now.getDate(),
      0,
      0,
      0,
      0,
    );
  }

  static evalTextLarge(text: string): string {
    if (text.length > 65535) {
      return text.substring(0, 65534);
    }
    return text;
  }

  static getCurrentSystemDate(): Date {
    return new Date();
  }

  static evalMonth(description: String): Date | null {
    const meses: { [key: string]: number } = {
      enero: 0,
      febrero: 1,
      marzo: 2,
      abril: 3,
      mayo: 4,
      junio: 5,
      julio: 6,
      agosto: 7,
      septiembre: 8,
      octubre: 9,
      noviembre: 10,
      diciembre: 11,
    };

    // Expresión regular para encontrar mes y año
    const regex = new RegExp(
      `(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre)\\s+(\\d{4})`,
      'i',
    );

    const match = description.match(regex);

    if (!match) {
      return null; // No se encontró mes y año
    }

    const mes = match[1].toLowerCase();
    const año = parseInt(match[2], 10);

    return new Date(año, meses[mes], 1);
  }
}
