import { Body, Controller, Get, Inject, Param, Put, Query } from '@nestjs/common';
import { ApiBody, ApiOkResponse, ApiParam, ApiQuery, ApiTags } from '@nestjs/swagger';
import { PrimeLogger, User } from 'src/framework';
import {
  AllowRoles,
  RoleShortNameEnum,
} from 'src/framework/infrastructure/decorators/allow-profiles/allow-roles.decorator';
import { ApiDefaultResponses } from 'src/framework/infrastructure/decorators/api-default-responses/api-default-responses.decorator';
import { PrimeUser } from 'src/framework/infrastructure/decorators/prime-user/prime-user.decorator';
import { AgileTenderFullInfoTO } from 'src/licitaapp/application';
import { TenderService } from 'src/licitaapp/application/service/tender-service/tender-service.interface';
import { UserCompanyAgileTenderService } from 'src/licitaapp/application/service/user-company-agile-tender-service/user-company-agile-tender-service.interface';
import { AgileTender, MinimalToEditAgileTender, UpdateMinimalToEditAgileTender } from 'src/licitaapp/domain';
import { PaginationAgileTenderTO } from 'src/licitaapp/domain/dto/pagination-agile-tender.to';

@Controller('agile-tender')
@ApiTags('agile-tender')
export class AgileTenderController {
  private readonly LOGGER = new PrimeLogger(AgileTenderController.name);
  constructor(
    @Inject('TenderService') private readonly tenderService: TenderService,
    @Inject('UserCompanyAgileTenderService')
    private readonly userCompanyAgileTenderService: UserCompanyAgileTenderService,
  ) {}

  @Get('admin-pagination/:page/:pageSize')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: [PaginationAgileTenderTO] })
  @ApiParam({ name: 'page', type: 'number' })
  @ApiParam({ name: 'pageSize', type: 'number' })
  @ApiQuery({ name: 'order', type: 'string', required: false })
  @ApiQuery({ name: 'filter', type: 'string', required: false })
  async adminPagination(
    @Param('page') page: number,
    @Param('pageSize') pageSize: number,
    @Query('order') order: string,
    @Query('filter') filter: string,
  ): Promise<PaginationAgileTenderTO[]> {
    this.LOGGER.warn(
      `Finding admin agile tenders page: ${page} pageSize: ${pageSize} order: ${order} filter: ${filter}`,
    );
    return await this.tenderService.getAgileTenderAdminPaginated(page, pageSize, order, filter);
  }

  @Get('by-id/:id')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: AgileTender })
  @ApiParam({ name: 'id', type: 'number' })
  async findById(
    @Param('id') id: number,
  ): Promise<AgileTender | null | undefined> {
    this.LOGGER.log(`Finding agile tender by id: ${id}`);
    return await this.tenderService.getAgileTenderById(id);
  }

  @Get('minimal-to-edit-by-id/:id')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: MinimalToEditAgileTender })
  @ApiParam({ name: 'id', type: 'number' })
  async findMinimalToEditById(
    @Param('id') id: number,
  ): Promise<MinimalToEditAgileTender | undefined> {
    this.LOGGER.log(`Finding agile tender by id: ${id}`);
    return await this.tenderService.getAgileTenderMinimalToEditById(id);
  }

  @Get(':page/:pageSize/:typeSearch/:companyId')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER)
  @ApiOkResponse({ type: [AgileTenderFullInfoTO] })
  @ApiParam({ name: 'page', type: 'number' })
  @ApiParam({ name: 'pageSize', type: 'number' })
  @ApiParam({ name: 'typeSearch', type: 'string' })
  @ApiParam({ name: 'companyId', type: 'number' })
  async paginationByUser(
    @Param('page') page: number,
    @Param('pageSize') pageSize: number,
    @Param('typeSearch') typeSearch: string,
    @Param('companyId') companyId: number,
    @PrimeUser() user: User
  ): Promise<AgileTenderFullInfoTO[]> {
    this.LOGGER.warn(
      `Finding agile tenders page: ${page} pageSize: ${pageSize} typeSearch: ${typeSearch} companyId: ${companyId}`,
    );
    return await this.tenderService.getAgileTenderPaginated(page, pageSize, typeSearch, companyId, user.id);
  }

  @Get('history-agile/:page/:pageSize/:companyId/:searchType/:isFavorite')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: [AgileTenderFullInfoTO] })
  @ApiParam({ name: 'page', type: 'number' })
  @ApiParam({ name: 'pageSize', type: 'number' })
  @ApiParam({ name: 'companyId', type: 'number' })
  @ApiParam({ name: 'searchType', type: 'string' })
  @ApiParam({ name: 'isFavorite', type: 'string' })
  async getTenderAgileHistoryPaginated(
    @PrimeUser() user: User,
    @Param('page') page: number,
    @Param('pageSize') pageSize: number,
    @Param('companyId') companyId: number,
    @Param('searchType') searchType: string,
    @Param('isFavorite') isFavorite: string,
  ): Promise<AgileTenderFullInfoTO[]> {
    this.LOGGER.log(
      `GetTenderAgilePaginated - user: ${user.id}, page: ${page}, pageSize: ${pageSize}, companyId: ${companyId}, searchType: ${searchType} isFavorite: ${isFavorite}`,
    );
    return await this.userCompanyAgileTenderService.getPaginatedAgileTenders(
      user.id,
      companyId,
      page,
      pageSize,
      isFavorite === 'true',
    );
  }

  @Put(
    'update-favorite-user-agile-tender/:agileTenderId/:companyId/:isFavorite/:agileCode',
  )
  @ApiDefaultResponses()
  @ApiOkResponse({ type: () => String })
  @ApiParam({ name: 'agileTenderId', type: 'number' })
  @ApiParam({ name: 'isFavorite', type: 'string' })
  @ApiParam({ name: 'companyId', type: 'number' })
  @ApiParam({ name: 'agileCode', type: 'string' })
  async updateFavoriteUserCompanyAgiletender(
    @PrimeUser() user: User,
    @Param('agileTenderId') agileTenderId: number,
    @Param('companyId') companyId: number,
    @Param('isFavorite') isFavorite: string,
    @Param('agileCode') agileCode: string,
  ): Promise<String> {
    this.LOGGER.log(
      `Favorite - user: ${user.id}, agileTenderId: ${agileTenderId}, isFavorite: ${isFavorite} companyId: ${companyId} agileCode: ${agileCode}`,
    );
    await this.userCompanyAgileTenderService.saveAll(
      user.id,
      companyId,
      [agileTenderId],
      isFavorite === 'true',
      agileCode,
      this.tenderService.getUserRequestService(),
    );
    return 'true';
  }

  @Get('favorites')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: [Number] })
  async listFavoriteAgileTender(@PrimeUser() user: User): Promise<number[]> {
    this.LOGGER.log(`ListFavoriteAgileTender - user: ${user.id}`);
    return await this.userCompanyAgileTenderService.getFavoriteList(user.id);
  }

  /*@Put('update-search-entities')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: String })
  @ApiBody({ type: AgileSearchEntityTO })
  async updateAgileTender(@Body() data: AgileSearchEntityTO): Promise<string> {
    this.LOGGER.log(`Updating agile tender with keywords: ${data.searchEntities}`);
    await this.tenderService.updateSearchEntitiesAgileTender(data);
    return 'Agile Tender updated successfully';
  }*/

  @Put('update-data-minimal-to-edit')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: String })
  @ApiBody({ type: UpdateMinimalToEditAgileTender })
  async updateMinimalAgileTender(@Body() dataAgiletender: UpdateMinimalToEditAgileTender): Promise<string> {
    await this.tenderService.updateMinimalAgileTender(dataAgiletender);
    return 'true';
  }
  
}
