import { Body, Controller, Delete, Get, Inject, Param, Put, Query } from '@nestjs/common';
import { ApiBody, ApiOkResponse, ApiParam, ApiQuery, ApiTags } from '@nestjs/swagger';
import { PrimeLogger, User } from 'src/framework';
import { AllowRoles, RoleShortNameEnum } from 'src/framework/infrastructure/decorators/allow-profiles/allow-roles.decorator';
import { ApiDefaultResponses } from 'src/framework/infrastructure/decorators/api-default-responses/api-default-responses.decorator';
import { PrimeUser } from 'src/framework/infrastructure/decorators/prime-user/prime-user.decorator';
import { CompanyService, KeywordService, UserCompanyService } from 'src/licitaapp';
import { CompanyKeywordService } from 'src/licitaapp/application/service/company-keyword-service/company-keyword-service.interface';
import { TenderService } from 'src/licitaapp/application/service/tender-service/tender-service.interface';
import { Company, CompanyFullTO, CompanyRegisterTO, MinimalCompany } from 'src/licitaapp/domain';
import { PaginationKeywordTO } from 'src/licitaapp/domain/dto/company.user.to';
import { UserPaginationTO } from 'src/licitaapp/domain/dto/user.pagination.to';

@ApiTags('company')
@Controller('company')
export class CompanyController {
  private readonly LOGGER = new PrimeLogger(CompanyController.name);
  constructor(
    @Inject('CompanyService') private readonly companyService: CompanyService,
    @Inject('UserCompanyService') private readonly userCompanyService: UserCompanyService,
    @Inject('TenderService') private readonly tenderService: TenderService,
    @Inject('CompanyKeywordService') private readonly companyKeywordService: CompanyKeywordService,
    @Inject('KeywordService') private readonly keywordService: KeywordService,
  ) {}

  @Get('companies-by-user')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER)
  @ApiOkResponse({ type: [Company] })
  async companiesByUserId(
    @PrimeUser() user: User,
  ): Promise<Company[]> {
    this.LOGGER.warn(`Finding if user: ${user.id} have any companies`);
    return await this.userCompanyService.findCompaniesByUserId(user.id);
  }

  @Put('check-tender/:id')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: () => CompanyFullTO })
  @ApiParam({ name: 'id', type: 'number' })
  async activeCheckTender(@Param('id') id: number): Promise<String> {
    this.LOGGER.warn(`ActiveCheckTender company ${id}`);
    this.companyService.updateCheckTenders(id, true);
    return 'true';
  }

  @Put(':id')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: () => CompanyFullTO })
  @ApiParam({ name: 'id', type: 'number' })
  @ApiBody({ type: CompanyRegisterTO })
  async update(
    @PrimeUser() user: User,
    @Param('id') id: number,
    @Body() companyData: CompanyRegisterTO,
  ): Promise<CompanyFullTO | null> {
    this.LOGGER.warn(`Updating company ${id}`);
    const outputObj = await this.companyService.updateCompany(id, user.id, companyData);
    this.tenderService.createDashboardCompany(id, [user.id]);
    return outputObj;
  }

  @Delete('company-keyword/:codeCategoriaMercadoPublico/:companyId')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: String })
  @AllowRoles(RoleShortNameEnum.USER)
  async logicalRemoveKeyword(
    @PrimeUser() user: User,
    @Param('codeCategoriaMercadoPublico') codeCategoriaMercadoPublico: number,
    @Param('companyId') companyId: number,
  ): Promise<String> {
    this.LOGGER.log(
      `remove join codeCategoriaMercadoPublico: ${codeCategoriaMercadoPublico} by user: ${user.id} companyid: ${companyId}`,
    );
    const keyword = await this.keywordService.findByCodeCategoriaMercadoPublico([codeCategoriaMercadoPublico]);
    await this.companyKeywordService.deleteByCompanyAndKeywordId(companyId, keyword[0].id);
    await this.companyService.updateCheckTenders(companyId, true);
    //this.tenderService.createDashboardCompany(companyId, [user.id]);
    return 'true';
  }

  @Get(':id')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER)
  @ApiParam({ name: 'id', type: 'number' })
  @ApiOkResponse({ type: () => CompanyFullTO })
  async findById(@Param('id') id: number): Promise<CompanyFullTO | null> {
    this.LOGGER.warn(`findById id ${id}`);
    return this.companyService.findByIdWithFetch(id);
  }

  @Get(':page/:pageSize')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER)
  @ApiOkResponse({ type: [Company] })
  @ApiParam({ name: 'page', type: 'number' })
  @ApiParam({ name: 'pageSize', type: 'number' })
  async paginationByUser(@PrimeUser() user: User, @Param('page') page: number, @Param('pageSize') pageSize: number ): Promise<Company[]> {
    this.LOGGER.warn(`paginationByUser user ${user.id} page ${page} pageSize ${pageSize}`);
    return await this.userCompanyService.paginationByUser(user.id, page, pageSize);
  }

  @Get()
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER)
  @ApiOkResponse({ type: [Company] })
  async findByUserId(@PrimeUser() user: User): Promise<Company[]> {
    this.LOGGER.warn(`findByUserId user ${user.id}`);
    return await this.userCompanyService.findCompaniesByUserId(user.id);
  }

  /*@Post('companies-keyword')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: [MinimalCompany] })
  @ApiBody({ type: KeywordCompanyTO })
  async confirmCompaniesKeywords(@Body() keywordCompanyDto: KeywordCompanyTO): Promise<MinimalCompany[]> {
    this.LOGGER.warn(`confirmCompaniesKeywords keywords ${keywordCompanyDto}`);
    return await this.companyService.findCompaniesKeywords(keywordCompanyDto);
  }*/

  @Delete('relation/:companyId/:userId')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: String })
  @AllowRoles(RoleShortNameEnum.USER)
  async logicalRemove(@PrimeUser() user: User, @Param('companyId') companyId: number, 
  @Param('userId') userId: number
): Promise<String> {
    this.LOGGER.warn(`delete companyId ${companyId} user ${userId}`);
    await this.userCompanyService.deleteByUserCompany(userId==0? user.id: userId, companyId);
    return 'true';
  }

  @Get('paginated-admin/:page/:pageSize')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: UserPaginationTO })
  @ApiParam({ name: 'page', type: 'number' })
  @ApiParam({ name: 'pageSize', type: 'number' })
  async paginationCompanyAdmin(
    @Param('page') page: number,
    @Param('pageSize') pageSize: number,
  ): Promise<UserPaginationTO> {
    this.LOGGER.warn(
      `paginationByCompanyAdmin page ${page} pageSize ${pageSize}`,
    );
    return await this.companyService.paginationByCompanyAdmin(page, pageSize);
  }

  @Get('paginated-keyword/:page/:pageSize')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: [PaginationKeywordTO] })
  @ApiParam({ name: 'page', type: 'number' })
  @ApiParam({ name: 'pageSize', type: 'number' })
  @ApiQuery({ name: 'socialReason', type: 'string', required: false })
  async paginationCompanyKeyword(
    @Param('page') page: number,
    @Param('pageSize') pageSize: number,
    @Query('socialReason') socialReason?: string
  ): Promise<PaginationKeywordTO[]> {
    this.LOGGER.warn(
      `paginationByCompanyKeyword page ${page} pageSize ${pageSize} socialReason ${socialReason}`,
    );
    return await this.companyService.paginationByCompanyKeyword(page, pageSize, socialReason);
  }

}
