import { Body, Controller, Delete, Get, Inject, Param, Post } from '@nestjs/common';
import { ApiBody, ApiOkResponse, ApiParam, ApiTags } from '@nestjs/swagger';
import { PrimeLogger, User } from 'src/framework';
import {
  AllowRoles,
  RoleShortNameEnum,
} from 'src/framework/infrastructure/decorators/allow-profiles/allow-roles.decorator';
import { ApiDefaultResponses } from 'src/framework/infrastructure/decorators/api-default-responses/api-default-responses.decorator';
import { PrimeUser } from 'src/framework/infrastructure/decorators/prime-user/prime-user.decorator';
import { CompanyService, KeywordService } from 'src/licitaapp';
import { CompanyKeywordService } from 'src/licitaapp/application/service/company-keyword-service/company-keyword-service.interface';
import { InsertKeyword, Keyword, KeywordItem } from 'src/licitaapp/domain';

@Controller('keyword')
@ApiTags('keyword')
export class KeywordController {
  private readonly LOGGER = new PrimeLogger(KeywordController.name);
  constructor(
    @Inject('KeywordService') private readonly keywordService: KeywordService,
    @Inject('CompanyKeywordService')
    private readonly companyKeywordService: CompanyKeywordService,
  ) {}

  @Get('find-admin-values')
  @ApiOkResponse({ type: [String] })
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  async findAdminValues(): Promise<string[]> {
    this.LOGGER.log(`findAdminValues`);
    return await this.keywordService.findAdminValues();
  }

  @Get(':search')
  @ApiOkResponse({ type: [KeywordItem] })
  @ApiDefaultResponses()
  @ApiParam({ name: 'search', type: 'string' })
  async findAllByName(@Param('search') search: string): Promise<KeywordItem[]> {
    return this.keywordService.findByName(search);
  }

  @Get('find-by-company/:companyId')
  @ApiOkResponse({ type: [Keyword] })
  @ApiDefaultResponses()
  @ApiParam({ name: 'companyId', type: 'number' })
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  async findByCompanyId(
    @Param('companyId') companyId: number,
  ): Promise<Keyword[]> {
    return this.companyKeywordService.findByCompanyId(companyId);
  }

  @Delete('errase-keyword-to-company/:keywordId/:companyId')
  @ApiDefaultResponses()
  @ApiOkResponse({ type: String })
  @AllowRoles(RoleShortNameEnum.USER)
  @ApiParam({ name: 'keywordId', type: 'number', required: true })
  @ApiParam({ name: 'companyId', type: 'number', required: true })
  async logicalRemove(
    @Param('keywordId') keywordId: number,
    @Param('companyId') companyId: number,
  ): Promise<String> {
    this.LOGGER.log(`logicalRemove keyword: ${keywordId} to company: ${companyId}`);
    await this.companyKeywordService.deleteByCompanyAndKeywordId(companyId, keywordId);
    return 'true';
  }

  @Post()
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiBody({ type: InsertKeyword })
  @ApiOkResponse({ type: String })
  async createKeyword(
    @Body() data: InsertKeyword,
  ) {
      this.LOGGER.warn(`createKeyword keyword: ${JSON.stringify(data)}`);
      await this.keywordService.save(data);
      return 'true';
  } 
}
