import {
  Body,
  Controller,
  Delete,
  Get,
  Inject,
  Param,
  Post,
  Put,
  Query,
} from '@nestjs/common';
import {
  ApiBody,
  ApiConflictResponse,
  ApiOkResponse,
  ApiParam,
  ApiQuery,
  ApiTags,
} from '@nestjs/swagger';
import { ApiErrorResponse, PrimeLogger } from 'src/framework';
import {
  AllowRoles,
  RoleShortNameEnum,
} from 'src/framework/infrastructure/decorators/allow-profiles/allow-roles.decorator';
import { ApiDefaultResponses } from 'src/framework/infrastructure/decorators/api-default-responses/api-default-responses.decorator';
import { MatchingWordService } from 'src/licitaapp/application/service/matching_word-service/matching_word-service.interface';
import {
  InsertMatchingWord,
  MatchingWord,
  UpdateMatchingWord,
} from 'src/licitaapp/domain';

@Controller('matching-word')
@ApiTags('matching-word')
export class MatchingWordController {
  private readonly LOGGER = new PrimeLogger(MatchingWordController.name);
  constructor(
    @Inject('MatchingWordService')
    private readonly matchingWordService: MatchingWordService,
  ) {}

  @Post()
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: String })
  @ApiBody({ type: InsertMatchingWord })
  async save(@Body() data: InsertMatchingWord): Promise<string> {
    this.LOGGER.log(`Saving new matching word: ${data}`);
    await this.matchingWordService.save(data);
    return 'true';
  }

  @Put()
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: String })
  @ApiBody({ type: UpdateMatchingWord })
  async update(@Body() data: UpdateMatchingWord): Promise<string> {
    this.LOGGER.log(`Updating matching word: ${data}`);
    await this.matchingWordService.update(data);
    return 'true';
  }

  @Get()
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
  @ApiOkResponse({ type: [MatchingWord] })
  async findAll(): Promise<MatchingWord[]> {
    this.LOGGER.log('Finding all matching words');
    return await this.matchingWordService.findAll();
  }

  @Get(':page/:pageSize')
  @ApiDefaultResponses()
  @AllowRoles(RoleShortNameEnum.USER)
  @ApiOkResponse({ type: [MatchingWord] })
  @ApiParam({ name: 'page', type: 'number'  })
  @ApiParam({ name: 'pageSize', type: 'number' })
  @ApiQuery({ name: 'subdivisionId', required: true, type: String, description: 'Filtro para titulos' })
  @ApiQuery({ name: 'wordSearch', required: false, type: String, description: 'Filtro para titulos' })
  async getPaginatedUserCampaign(@Param('page') page: number, 
  @Param('pageSize') pageSize: number, @Query('subdivisionId') subdivisionId: string, @Query('wordSearch') wordSearch?: string): Promise<MatchingWord[]> {
      this.LOGGER.log(`getPaginatedUserCampaign - page: ${page}, pageSize: ${pageSize}, subdivisionId: ${subdivisionId} wordSearch: ${wordSearch}`);
      return await this.matchingWordService.getPagination(page, pageSize, subdivisionId, wordSearch ? wordSearch : '');
  }

  @Delete(':id')
  @ApiDefaultResponses()
  @ApiConflictResponse({ type: ApiErrorResponse })
  @ApiParam({ name: 'id', type: 'number' })
  @ApiOkResponse({ type: String })
  async erraseAccount(@Param('id') id: number): Promise<String> {
    this.LOGGER.log(`erraseAccount user ${id}`);
    await this.matchingWordService.delete(id);
    return 'true';
  }
}
