import { Injectable } from '@nestjs/common';
import { and, eq } from 'drizzle-orm';
import { PrimeLogger, schema, TxType } from 'src/framework';
import { DBConfigService } from 'src/framework/infrastructure/drizzle/drizzle.provider';
import { CompanyKeywordRepository } from 'src/licitaapp/application/repository/company-keyword-repository/company-keyword-repository.interface';
import { Keyword } from 'src/licitaapp/domain';

@Injectable()
export class CompanyKeywordRepositoryImpl implements CompanyKeywordRepository {
    private readonly LOGGER = new PrimeLogger(CompanyKeywordRepositoryImpl.name);
    constructor(private readonly db: DBConfigService) {}
   async findByCompanyId(companyId: number, tx?: TxType): Promise<Keyword[]> {
    this.LOGGER.log(`findByCompanyId companyId: ${companyId}`);
        return await (tx || this.db.conn)
      .select({
        id: schema.keywordTable.id,
        value: schema.keywordTable.value,
        origin: schema.keywordTable.origin,
        metadata: schema.keywordTable.metadata,
        codeCategoriaMercadoPublico: schema.keywordTable.codeCategoriaMercadoPublico,
      })
      .from(schema.keywordTable)
      .leftJoin(schema.companyKeywordTable, eq(schema.keywordTable.id, schema.companyKeywordTable.keywordId))
      .where(and(
        eq(schema.companyKeywordTable.companyId, companyId),
        eq(schema.keywordTable.active, true)
        ))
      .then((rows) => {
        return rows.map((row) => new Keyword(row.id, row.value, row.codeCategoriaMercadoPublico, row.origin, undefined, row.metadata));
      });
    }
    deleteByCompanyAndKeywordId(companyId: number, keywordId: number, tx?: TxType): Promise<void> {
        this.LOGGER.log(`deleteByCompanyAndKeywordId companyId: ${companyId} keywordId: ${keywordId}`);
        (tx || this.db.conn)
        .delete(schema.companyKeywordTable)
        .where(
            and(
                eq(schema.companyKeywordTable.companyId, companyId),
                eq(schema.companyKeywordTable.keywordId, keywordId)
            )
        ).execute();
        return Promise.resolve();
    }
    async deleteByCompanyId(companyId: number, tx?: TxType): Promise<void> {
         this.LOGGER.log(`deleteByCompanyId keyword: ${companyId}`);
        await (tx || this.db.conn)
        .delete(schema.companyKeywordTable)
        .where(eq(schema.companyKeywordTable.companyId, companyId));
    }
    async saveAll(companyId: number, keywordId: number[], tx?: TxType): Promise<void> {
        this.LOGGER.log(`saveAll companyId: ${companyId} keywordId: ${keywordId}`);

        for (const keywordIdItem of keywordId) {
            try {
            const validated = schema.companyKeywordTableInsertSchema.parse({
                companyId: +companyId,
                keywordId: +keywordIdItem
            });
            
            await (tx || this.db.conn)
                .insert(schema.companyKeywordTable)
                .values(validated);
            } catch (error) {
                // Log duplicate key error and continue with next item
                if (error.code === '23505' || error.message?.includes('duplicate')) {
                    this.LOGGER.warn(`Duplicate key ignored for companyId: ${companyId}, keywordId: ${keywordIdItem}`);
                } else {
                    this.LOGGER.error(`Error inserting companyId: ${companyId}, keywordId: ${keywordIdItem}`, error);
                    throw error; // Re-throw non-duplicate errors
                }
            }
        }       
    
    }
}
