import { forwardRef, Inject, Injectable } from '@nestjs/common';
import {
  PrimeLogger,
  TxType,
} from 'src/framework';
import { DBConfigService } from 'src/framework/infrastructure/drizzle/drizzle.provider';
import { CompanyRepository } from 'src/licitaapp/application';
import {
  CompanyRegisterTO,
  Company,
  InsertCompany,
  MinimalCompany,
} from 'src/licitaapp/domain';
import { schema } from 'src/framework';
import { inspect } from 'util';
import { and, eq, inArray, like } from 'drizzle-orm';
import { CompanyUserTO, PaginationKeywordTO } from 'src/licitaapp/domain/dto/company.user.to';
import { TenderUtil } from 'src/licitaapp/domain/util';
import { Subdivision } from 'src/framework/domain/entities/subdivision.entity';

@Injectable()
export class CompanyRepositoryImpl implements CompanyRepository {
  private readonly LOGGER = new PrimeLogger(CompanyRepositoryImpl.name);
  constructor(@Inject(forwardRef(() => DBConfigService)) private readonly db: DBConfigService) {}
  /*TODO: reparar async findCompaniesKeywords(keywordIds: number[]): Promise<MinimalCompany[]> {
    this.LOGGER.log(`findCompaniesKeywords repository: ${inspect(keywordIds)}`);
    return await this.db.conn
      .select({
        id: schema.companyTable.id,
        dni: schema.companyTable.dni,
        socialReason: schema.companyTable.socialReason,
      })
      .from(schema.companyTable)
      .leftJoin(
        schema.keywordTable,
        eq(
          schema.companyTable.id,
          schema.keywordTable.companyId,
        ),
      )
      .where(inArray(schema.keywordTable.id, keywordIds))
      .then((rows) => {
        if (rows.length === 0) {
          return [];
        }
        return rows.map(
          (row) => new MinimalCompany(row.id, row.dni, row.socialReason),
        );
      });
  }*/
  async paginationByCompanyKeyword(page: number, pageSize: number, socialReason?: string): Promise<PaginationKeywordTO[]> {
    this.LOGGER.log(`pagination admin page ${page} pageSize ${pageSize}`);
    const offset = (page - 1) * pageSize;
    const baseQuery = this.db.conn
      .select({
        companyId: schema.companyTable.id,
        dni: schema.companyTable.dni,
        socialReason: schema.companyTable.socialReason,
        active: schema.companyTable.active,
      })
      .from(schema.companyTable);

    const query = socialReason
      ? baseQuery.where(like(schema.companyTable.socialReason, '%' + socialReason + '%'))
      : baseQuery;

    return await query
      .offset(offset)
      .limit(Number(pageSize))
      .then((rows) => {
        if (rows.length === 0) {
          return [];
        }
        return rows.map((row) => {
          const company = new PaginationKeywordTO();
          company.companyId = row.companyId;
          company.dni = row.dni;
          company.active = row.active;
          company.socialReason = row.socialReason;
          company.amountTenders = 0;
          company.amountAgileTenders = 0;
          return company;
        });
      });
  }
  async getAllCompanyWithoutKeywords(): Promise<number[]> {
    this.LOGGER.log('Getting all company without keywords');
    var outputList: number[] = [];
    await this.db.conn.execute(
      `SELECT c.id AS company_id, COUNT(k.id) AS total_keywords FROM company c JOIN keyword k ON k.company_id = c.id WHERE c.active = true and k.active = true GROUP BY c.id HAVING COUNT(k.id) = 1`
    ).then((rows:any) => {
      const items = rows[0];
      outputList = items.map((item: any) => {
        return item.company_id;
      });
    })
    .catch((err) => {
      this.LOGGER.error(`Error getting companies without keywords: ${inspect(err)}`);
      throw err;
    });
    return outputList;
  }

  async getCompanyIdsActive(): Promise<number[]> {
    this.LOGGER.log(`getCompanyIdsToRecalculateTender`);
    return await this.db.conn
      .select({
        id: schema.companyTable.id,
      })
      .from(schema.companyTable)
      .where(and(eq(schema.companyTable.active, true)))
      .then((rows) => {
        if (rows.length === 0) {
          return [];
        }
        return rows.map(
          (row) => row.id,
        );
      });
  }
  async updateCheckTenders(companyId: number, checkTender: boolean, tx?: TxType): Promise<boolean> {
    this.LOGGER.log(`updateCheckTenders: ${companyId}`);
    await (tx || this.db.conn)
      .update(schema.companyTable)
      .set({
        updatedAt: TenderUtil.getCurrentSystemDate(),
        checkTender: checkTender,
       })
      .where(eq(schema.companyTable.id, companyId))
      .execute();
    return true;
  }

  async update(companyId: number, companyData: CompanyRegisterTO, tx?: TxType): Promise<boolean> {
    this.LOGGER.log(`update repository: ${companyId}`);
    const validatedCompany = schema.companyTableUpdateSchema.parse({
      dni: companyData.company.dni,
      socialReason: companyData.company.socialReason,
      updatedAt: TenderUtil.getCurrentSystemDate(),
      checkTender: true,
    });
    await (tx || this.db.conn)
      .update(schema.companyTable)
      .set(validatedCompany)
      .where(eq(schema.companyTable.id, companyId))
      .execute();
    return true;
  }
  
  async save(company: InsertCompany, tx?: TxType): Promise<Company> {
    this.LOGGER.log(`save repository: ${inspect(company)}`);
    const companyFound = await this.findByDNI(company.dni, tx);
    if (companyFound) {
      this.LOGGER.log(`Company found with id: ${companyFound.id}`);
      return companyFound;
    }
    const validatedCompany = schema.companyTableInsertSchema.parse({
      dni: company.dni,
      socialReason: company.socialReason,
    });
    const insertedCompanyId = await (tx || this.db.conn)
      .insert(schema.companyTable)
      .values(validatedCompany)
      .$returningId()
      .then((rows) => {
        return rows[0].id;
      });

    return await this.findById(insertedCompanyId, tx).then((dataObject) => {
      if (dataObject) {
        return dataObject;
      } else {
        throw new Error('Company not found');
      }
    });
  }

  async saveCompanySubdivisions(
    companyId: number,
    listSubdivisions: Subdivision[],
    tx?: TxType,
  ): Promise<void> {
    listSubdivisions = listSubdivisions.filter((subdivision) => subdivision.id !== 7777);
    this.LOGGER.log(`saveCompanySubdivisions repository: ${companyId}`);
    await (tx || this.db.conn)
      .insert(schema.companySubdivisionTable)
      .values(
        listSubdivisions.map((subdivision) => {
          return {
            companyId,
            subdivisionId: subdivision.id,
          };
        }),
      )
      .execute();
  }

  async findCompaniesSubdivisions(companyId: number, tx?: TxType): Promise<Subdivision[]> {
    this.LOGGER.log(`findCompaniesSubdivisions repository: ${companyId}`);
    return await (tx || this.db.conn)
      .select({
        id: schema.subdivisionTable.id,
        name: schema.subdivisionTable.name,
        code: schema.subdivisionTable.code,
      })
      .from(schema.companySubdivisionTable)
      .innerJoin(
        schema.subdivisionTable,
        eq(
          schema.companySubdivisionTable.subdivisionId,
          schema.subdivisionTable.id,
        ),
      )
      .where(eq(schema.companySubdivisionTable.companyId, companyId))
      .then((rows) => {
        if (rows.length === 0) {
          return [];
        }
        return rows.map(
          (row) => new Subdivision(row.id, row.name, row.code),
        );
      });
  }

  async deletedCompanySubdivisions(companyId: number, tx?: TxType): Promise<void> {
    this.LOGGER.log(`deletedCompanySubdivisions repository: ${companyId}`);
    await (tx || this.db.conn)
      .delete(schema.companySubdivisionTable)
      .where(eq(schema.companySubdivisionTable.companyId, companyId))
      .execute();
  }

  async findByDNI(dni: string, tx?: TxType): Promise<Company | null> {
    this.LOGGER.log(`findByDNI repository: ${dni}`);
    return await (tx || this.db.conn)
      .select({
        id: schema.companyTable.id,
        dni: schema.companyTable.dni,
        socialReason: schema.companyTable.socialReason,
      })
      .from(schema.companyTable)
      .where(eq(schema.companyTable.dni, dni))
      .then((rows) => {
        if (rows.length === 0) {
          return null;
        }
        return new Company(rows[0].id, rows[0].dni, rows[0].socialReason);
      });
  }

  async findById(id: number, tx?: TxType): Promise<Company | null> {
    this.LOGGER.log(`findById repository: ${id}`);
    return await (tx || this.db.conn)
      .select({
        id: schema.companyTable.id,
        dni: schema.companyTable.dni,
        socialReason: schema.companyTable.socialReason,
      })
      .from(schema.companyTable)
      .where(eq(schema.companyTable.id, id))
      .then((rows) => {
        if (rows.length === 0) {
          return null;
        }

        return new Company(rows[0].id, rows[0].dni, rows[0].socialReason);
      });
  }

  async getSubdivisionsByCompanyId(companyId: number): Promise<Subdivision[]> {
    this.LOGGER.log(`getSubdivisionsByCompanyId repository: ${companyId}`);
    return await this.db.conn
      .select({
        id: schema.subdivisionTable.id,
        name: schema.subdivisionTable.name,
        code: schema.subdivisionTable.code,
      })
      .from(schema.companySubdivisionTable)
      .innerJoin(
        schema.subdivisionTable,
        eq(
          schema.companySubdivisionTable.subdivisionId,
          schema.subdivisionTable.id,
        ),
      )
      .where(eq(schema.companySubdivisionTable.companyId, companyId))
      .then((rows) => {
        if (rows.length === 0) {
          return [];
        }
        return rows.map(
          (row) => new Subdivision(row.id, row.code, row.name),
        );
      });
  }

  async getCompanyIdsToRecalculateTender(): Promise<number[]> {
    this.LOGGER.log(`getCompanyIdsToRecalculateTender`);
    return await this.db.conn
      .select({
        id: schema.companyTable.id,
      })
      .from(schema.companyTable)
      .where(and(eq(schema.companyTable.active, true), eq(schema.companyTable.checkTender, true)))
      .then((rows) => {
        if (rows.length === 0) {
          return [];
        }
        return rows.map(
          (row) => row.id,
        );
      });
  }

  async paginationByCompanyAdmin(
    page: number,
    pageSize: number,
  ): Promise<CompanyUserTO[]> {
    this.LOGGER.log(`pagination admin page ${page} pageSize ${pageSize}`);
    const offset = (page - 1) * pageSize;
    return await this.db.conn.
      select({
        companyId: schema.companyTable.id, 
        dni: schema.companyTable.dni, 
        socialReason: schema.companyTable.socialReason,
        active: schema.companyTable.active,
      })
      .from(schema.companyTable)
      //.where(eq(schema.companyTable.active, true))
      .orderBy(schema.companyTable.dni)
      .offset(offset)
      .limit(Number(pageSize))
      .then((rows) => {
        if (rows.length === 0) {
          return [];
        }
        return rows.map(
          (row) =>{
            let company = new CompanyUserTO();
            company.companyId = row.companyId;
            company.dni = row.dni;
            company.active = row.active;
            company.socialReason = row.socialReason;
            company.amountTenders = 0;
            return company;
          }
        );
      });
  }
}
