import { Injectable } from '@nestjs/common';
import { and, eq, inArray, isNull, sql } from 'drizzle-orm';
import { PrimeLogger, schema, TxType } from 'src/framework';
import { DBConfigService } from 'src/framework/infrastructure/drizzle/drizzle.provider';
import { KeywordRepository } from 'src/licitaapp/application';
import { Keyword, InsertKeyword, Metadata, KeywordOrigin } from 'src/licitaapp/domain';
import { TenderUtil } from 'src/licitaapp/domain/util';

@Injectable()
export class KeywordRepositoryImpl implements KeywordRepository {
  private readonly LOGGER = new PrimeLogger(KeywordRepositoryImpl.name);
  constructor(private readonly db: DBConfigService) {}
  async findByCodeCategoriaMercadoPublico(codeCategoriaMercadoPublico: number[], tx?: TxType): Promise<Keyword[]> {
    this.LOGGER.log(`findByCodeCategoriaMercadoPublico codeCategoriaMercadoPublico: ${codeCategoriaMercadoPublico}`);
    return await ( tx || this.db.conn)
      .select({
        id: schema.keywordTable.id,
        codeCategoriaMercadoPublico: schema.keywordTable.codeCategoriaMercadoPublico,
      })
      .from(schema.keywordTable)
      .where(inArray(schema.keywordTable.codeCategoriaMercadoPublico, codeCategoriaMercadoPublico))
      .then((rows) => {
        return rows.map((row) => {
          return new Keyword(row.id, '', row.codeCategoriaMercadoPublico, KeywordOrigin.MERCADO_PUBLICO);
        });
      }
    );
  }
  async findKeywordsByNameList(names: string[]): Promise<number[]> {
    this.LOGGER.log(`findKeywordsByNameList names: ${names}`);
    return await (this.db.conn)
      .select({
        id: schema.keywordTable.id,
      })
      .from(schema.keywordTable)
      .where(and(eq(schema.keywordTable.active, true), inArray(schema.keywordTable.value, names)))
      .then((rows) => {
        return rows.map((row) => row.id);
      }
    );
  }
  async findAdminValues(): Promise<string[]> {
    this.LOGGER.log(`findAdminValues`);
    return await (this.db.conn)
      .select({
        value: schema.keywordTable.value,
      })
      .from(schema.keywordTable)
      .where(and(eq(schema.keywordTable.active, true), eq(schema.keywordTable.origin, KeywordOrigin.ADMIN)))
      .groupBy(schema.keywordTable.value)
      .then((rows) => {
        return rows.map((row) => row.value);
      }
    );
  }
  async getAllWithouthMetadata(tx?: TxType): Promise<Keyword[]> {
    this.LOGGER.log(`getAllWithouthMetadata`);
    return await (tx || this.db.conn)
      .select({
        id: schema.keywordTable.id,
        value: schema.keywordTable.value,
        origin: schema.keywordTable.origin,
        codeCategoriaMercadoPublico: schema.keywordTable.codeCategoriaMercadoPublico,
      })
      .from(schema.keywordTable)
      .where(and(eq(schema.keywordTable.active, true), isNull(schema.keywordTable.metadata)))
      .then((rows) => {
        return rows.map((row) => new Keyword(row.id, row.value, row.codeCategoriaMercadoPublico, row.origin));
      }
    );
  }
  /*async findByCompanyId(companyId: number, isNullMetadata: boolean, tx?: TxType): Promise<Keyword[]> {
    const baseWhereClause = and(
      eq(schema.keywordTable.companyId, companyId),
      eq(schema.keywordTable.active, true)
    );

    const metadataClause = isNullMetadata  && isNotNull(schema.keywordTable.metadata);

    return await (tx || this.db.conn)
      .select({
        id: schema.keywordTable.id,
        value: schema.keywordTable.value,
        origin: schema.keywordTable.origin,
        metadata: schema.keywordTable.metadata,
        codeCategoriaMercadoPublico: schema.keywordTable.codeCategoriaMercadoPublico,
      })
      .from(schema.keywordTable)
      .where(metadataClause ? and(baseWhereClause, metadataClause): and(baseWhereClause))
      .then((rows) => {
        return rows.map((row) => new Keyword(row.id, row.value, row.codeCategoriaMercadoPublico, row.origin, undefined, undefined, row.metadata));
      });
  }*/

  /*async findByCompanyLimitedId(companyId: number, tx?: TxType): Promise<Keyword[]> {
    return await (tx || this.db.conn)
      .select({
        id: schema.keywordTable.id,
        value: schema.keywordTable.value,
        origin: schema.keywordTable.origin,
        codeCategoriaMercadoPublico: schema.keywordTable.codeCategoriaMercadoPublico,
      })
      .from(schema.keywordTable)
      .where(and(eq(schema.keywordTable.companyId, companyId), eq(schema.keywordTable.active, true)))
      .then((rows) => {
        return rows.map((row) => new Keyword(row.id, row.value, row.codeCategoriaMercadoPublico, row.origin, undefined, undefined, null));
      });
  }*/

  async save(keyword: InsertKeyword, tx?: TxType): Promise<Keyword> {
    this.LOGGER.log(`save keyword: ${JSON.stringify(keyword)}`);
    const validatedKeyword = schema.keywordTableInsertSchema.parse({
      value: keyword.value,
      origin: keyword.origin,
      active: true,
      codeCategoriaMercadoPublico: keyword.codeCategoriaMercadoPublico,
    });

    await (tx || this.db.conn).execute(
        sql.raw(`INSERT INTO keyword (active, value, origin, code_categoria_mercado_publico) VALUES 
          (true, '${keyword.value}', '${keyword.origin}', ${keyword.codeCategoriaMercadoPublico});`)            
      )
      .catch((err) => {
        this.LOGGER.error(`Error fetching registered users: ${err}`);
        throw new Error(`Error fetching registered users: ${err}`);
      });
    return new Keyword(0, validatedKeyword.value, validatedKeyword.codeCategoriaMercadoPublico, validatedKeyword.origin, undefined);
  }

  async saveAll(keywords: InsertKeyword[], tx?: TxType): Promise<Keyword[]> {
    if(keywords.length === 0){
      return [];
    }
    const validatedKeywords = keywords.map((keyword) =>
      schema.keywordTableInsertSchema.parse({
        codeCategoriaMercadoPublico: keyword.codeCategoriaMercadoPublico,
        value: keyword.value,
        origin: keyword.origin,
        active: true,
        //metadata: keyword.metadata as Metadata ?? undefined,
      }),
    );
    const insertedKeywordIds = await (tx || this.db.conn)
      .insert(schema.keywordTable)
      .values(validatedKeywords)
      .$returningId()
      .then((rows) => {
        return rows.map((row) => row.id);
      });

    return await Promise.all(
      insertedKeywordIds.map((id) => this.findById(id)),
    ).then((keywords) => {
      return keywords.filter((keyword) => keyword !== null) as Keyword[];
    });
  }

  async updateMetadata(keywordId: number, metadata: Metadata): Promise<void> {
    this.LOGGER.log(`updateMetadata keyword: ${keywordId}`);
    
    await (this.db.conn)
      .update(schema.keywordTable)
      .set({ 
        updatedAt: TenderUtil.getCurrentSystemDate(), 
        metadata: metadata, 
      })
      .where(eq(schema.keywordTable.id, keywordId));

  }

  async findById(id: number, tx?: TxType): Promise<Keyword | null | undefined> {
    return await (tx || this.db.conn)
      .select({
        id: schema.keywordTable.id,
        value: schema.keywordTable.value,
        origin: schema.keywordTable.origin,
        codeCategoriaMercadoPublico: schema.keywordTable.codeCategoriaMercadoPublico,
      })
      .from(schema.keywordTable)
      .where(eq(schema.keywordTable.id, id))
      .then((rows) => {
        if (rows.length === 0) {
          return null;
        }

        return new Keyword(rows[0].id, rows[0].value, rows[0].codeCategoriaMercadoPublico, rows[0].origin);
      });
  }

  async logicalRemove(id: number, tx?: TxType): Promise<void> {
    this.LOGGER.log(`logicalRemove keyword: ${id}`);
    await (tx || this.db.conn)
      .update(schema.keywordTable)
      .set({ active: false, deletedAt: TenderUtil.getCurrentSystemDate() })
      .where(eq(schema.keywordTable.id, id));
  }

  /*async deleteByCompanyId(companyId: number, tx?: TxType): Promise<void> {
    this.LOGGER.log(`deleteByCompanyId keyword: ${companyId}`);
    await (tx || this.db.conn)
      .delete(schema.keywordTable)
      .where(eq(schema.keywordTable.companyId, companyId));
  }*/
}
