import { Injectable } from '@nestjs/common';
import { and, eq, like, or } from 'drizzle-orm';
import { PrimeLogger, schema, TxType } from 'src/framework';
import { Subdivision } from 'src/framework/domain/entities/subdivision.entity';
import { DBConfigService } from 'src/framework/infrastructure/drizzle/drizzle.provider';
import { MatchingWordRepository } from 'src/licitaapp/application/repository/matching_word-repository/matching_word-repository.interface';
import { InsertMatchingWord, MatchingWord, UpdateMatchingWord } from 'src/licitaapp/domain';
import { TenderUtil } from 'src/licitaapp/domain/util';

@Injectable()
export class MatchingWordRepositoryImpl implements MatchingWordRepository {
  private readonly LOGGER = new PrimeLogger(MatchingWordRepositoryImpl.name);
  constructor(private readonly db: DBConfigService) {}
  async getPagination(page: number, pageSize: number, subdivisionId?: string, wordSearch?: string): Promise<MatchingWord[]> {
    this.LOGGER.log(`pagination - page: ${page}, pageSize: ${pageSize}, subdivisionId: ${subdivisionId}, wordSearch: ${wordSearch}`);
      const offset = (page - 1) * pageSize;
      const selectFields = {        
        id: schema.matchingWordTable.id,
        coincidence: schema.matchingWordTable.coincidence,
        subdivisionId: schema.matchingWordTable.subdivisionId,
        subdivision: schema.subdivisionTable
      };
        let query = this.db.conn
          .select(selectFields)
          .from(schema.matchingWordTable)
          .leftJoin(schema.subdivisionTable, eq(schema.matchingWordTable.subdivisionId, schema.subdivisionTable.id))
          .where(
            or(
              and(
              eq(schema.matchingWordTable.active, true),
              eq(schema.matchingWordTable.subdivisionId, +subdivisionId!),
              ...(wordSearch ? [like(schema.matchingWordTable.coincidence, `%${wordSearch}%`)] : [])
            ),
            and(
              eq(schema.matchingWordTable.active, true),
              eq(schema.matchingWordTable.subdivisionId, +subdivisionId!),
              ...(wordSearch ? [like(schema.matchingWordTable.coincidence, `%${wordSearch.toUpperCase()}%`)] : [])
            )
            ),
          )
          .limit(Number(pageSize))
          .offset(offset);

        const rows = await query;

        if (rows.length === 0) {
          return [];
        }

        return rows.map(
          (row) =>{
            return this.mapRowToEntity(row);
          }
        );
  }
  mapRowToEntity(rowRecord: any): MatchingWord {
    return new MatchingWord(
      rowRecord.id,
      rowRecord.coincidence,
      rowRecord.subdivision? rowRecord.subdivision : null,
      rowRecord.subdivisionId,
    );
  }
  findAll(): Promise<MatchingWord[]> {
    this.LOGGER.log('Finding all matching words');
    return this.db.conn
      .select({
        id: schema.matchingWordTable.id,
        coincidence: schema.matchingWordTable.coincidence,
        subdivisionId: schema.matchingWordTable.subdivisionId,
      })
      .from(schema.matchingWordTable)
      .where(eq(schema.matchingWordTable.active, true))
      .then((rows) => rows.map((row) => this.mapRowToEntity(row)));
  }
  async findById(matchingWordId: number, tx?: TxType): Promise<MatchingWord | null | undefined> {
    this.LOGGER.log(`Finding matching word with ID: ${matchingWordId}`);
    return await (tx || this.db.conn)
          .select({
            id: schema.matchingWordTable.id,
            coincidence: schema.matchingWordTable.coincidence,
            subdivisionId: schema.matchingWordTable.subdivisionId,
          })
          .from(schema.matchingWordTable)
          .where(eq(schema.matchingWordTable.id, matchingWordId))
          .then((rows) => {
            if (rows.length === 0) {
              return null;
            }
            return this.mapRowToEntity(rows[0]);
          });
  }
  async save(matchingWord: InsertMatchingWord, tx?: TxType): Promise<boolean> {
    this.LOGGER.log(`Saving matching word: ${JSON.stringify(matchingWord)}`);
    const validatedMatchingWord = schema.matchingWordTableInsertSchema.parse({
        coincidence: matchingWord.coincidence,
        subdivisionId: matchingWord.subdivisionId,
    });
    return (tx || this.db.conn)
      .insert(schema.matchingWordTable)
      .values(validatedMatchingWord)
      .then(() => true);
  }
  async update(
    matchingWord: UpdateMatchingWord,
    tx?: TxType,
  ): Promise<boolean> {
    this.LOGGER.log(`Updating matching word with ID: ${matchingWord.id}`);
    const validatedMatchingWord = schema.matchingWordTableUpdateSchema.parse({
        subdivisionId: matchingWord.subdivisionId,
        coincidence: matchingWord.coincidence,
        updatedAt: TenderUtil.getCurrentSystemDate(),
    });
    await this.db.conn
      .update(schema.matchingWordTable)
      .set(validatedMatchingWord)
      .where(eq(schema.matchingWordTable.id, matchingWord.id))
      .execute();
    return true;
  }
  
  async delete(matchingWordId: number, tx?: TxType): Promise<boolean> {
    await (this.db.conn)
      .update(schema.matchingWordTable)
      .set({ 
        active: false,
        deletedAt: TenderUtil.getCurrentSystemDate(),
      })
      .where(eq(schema.matchingWordTable.id, matchingWordId));
    return true;
  }
}
