import { Injectable } from '@nestjs/common';
import { and, eq, inArray, isNull, like } from 'drizzle-orm';
import { PrimeLogger, schema } from 'src/framework';
import { DBConfigService } from 'src/framework/infrastructure/drizzle/drizzle.provider';
import { TenderKeywordCategoryRepository } from 'src/licitaapp/application/repository/tender-keyword-category-repository/tender-keyword-category-repository.interface';
import { InsertTenderKeywordCategory, TenderKeywordCategory } from 'src/licitaapp/domain/entities/tender-keyword-category.entity';
import { TenderUtil } from 'src/licitaapp/domain/util';

@Injectable()
export class TenderKeywordCategoryRepositoryImpl implements TenderKeywordCategoryRepository {
    private readonly LOGGER = new PrimeLogger(TenderKeywordCategoryRepositoryImpl.name);
    constructor(private readonly db: DBConfigService) {}
    async findAllByWordFilter(wordFilter: string, returnNormalTender: boolean): Promise<number[]> {
        this.LOGGER.log(`findAllByWordFilter wordFilter: ${wordFilter} returnNormalTender: ${returnNormalTender}`);
        const whereSql = returnNormalTender ? 
            and(
                isNull(schema.tenderKeywordCategoryTable.agileTenderId),
                eq(schema.tenderKeywordCategoryTable.active, true),
                like(schema.tenderKeywordCategoryTable.itemDescription, `%${wordFilter}%`)
            ) :
            and(
                isNull(schema.tenderKeywordCategoryTable.normalTenderId),
                eq(schema.tenderKeywordCategoryTable.active, true),
                like(schema.tenderKeywordCategoryTable.itemDescription, `%${wordFilter}%`)
            );
        return await (this.db.conn)
            .select({
                agileTenderId: schema.tenderKeywordCategoryTable.agileTenderId,
                normalTenderId: schema.tenderKeywordCategoryTable.normalTenderId,
            })
            .from(schema.tenderKeywordCategoryTable)
            .where(whereSql).then((rows) => {
                return rows.map((row: any) => returnNormalTender ? row.normalTenderId : row.agileTenderId);
            });
    }
    async findAllByCodeMercadoPublico(codeMercadoPublico: number[], returnNormalTender: boolean): Promise<number[]> {
        this.LOGGER.log(`findAllByCodeMercadoPublico codeMercadoPublico: ${codeMercadoPublico} returnNormalTender: ${returnNormalTender}`);
        const whereSql = returnNormalTender ? 
            and(
                isNull(schema.tenderKeywordCategoryTable.agileTenderId),
                eq(schema.tenderKeywordCategoryTable.active, true),
                inArray(schema.tenderKeywordCategoryTable.codeMercadoPublico, codeMercadoPublico)
            ) :
            and(
                isNull(schema.tenderKeywordCategoryTable.normalTenderId),
                eq(schema.tenderKeywordCategoryTable.active, true),
                inArray(schema.tenderKeywordCategoryTable.codeMercadoPublico, codeMercadoPublico)
            );
        return await (this.db.conn)
            .select({
                agileTenderId: schema.tenderKeywordCategoryTable.agileTenderId,
                normalTenderId: schema.tenderKeywordCategoryTable.normalTenderId,
            })
            .from(schema.tenderKeywordCategoryTable)
            .where(whereSql).then((rows) => {
                return rows.map((row: any) => returnNormalTender ? row.normalTenderId : row.agileTenderId);
            });
    }
    async saveAgileTenderCategory(tenderId: number, codeMercadoPublico: string, itemDescription: string): Promise<void> {
        //for (const code of codeMercadoPublico) {
            this.LOGGER.log(`saveAgileTenderCategory tenderId: ${tenderId} codeMercadoPublico: ${codeMercadoPublico}`);
            await this.upsert({
                agileTenderId: tenderId,
                codeMercadoPublico: parseInt(codeMercadoPublico),
                itemDescription: itemDescription,
            });
        //}
    }
    async saveNormalTenderCategory(tenderId: number, codeMercadoPublico: string, itemDescription: string): Promise<void> {
        //for (const code of codeMercadoPublico) {
            this.LOGGER.log(`saveNormalTenderCategory tenderId: ${tenderId} codeMercadoPublico: ${codeMercadoPublico}`);
            await this.upsert({
                normalTenderId: tenderId,
                codeMercadoPublico: parseInt(codeMercadoPublico),
                itemDescription: itemDescription,
            });
       // }
        return Promise.resolve();
    }
    async errase(tenderKeywordCategoryId: number): Promise<void> {
        this.LOGGER.log(`delete tenderKeywordCategoryId: ${tenderKeywordCategoryId}`);
        await (this.db.conn)
            .delete(schema.tenderKeywordCategoryTable)
            .where(eq(schema.tenderKeywordCategoryTable.id, tenderKeywordCategoryId)).execute();
    }
    async findAllByNormaltenderId(normalTenderId: number): Promise<TenderKeywordCategory[]> {
        this.LOGGER.log(`findByNormalTenderId normalTenderId: ${normalTenderId}`);
        return await (this.db.conn)
            .select({
                id: schema.tenderKeywordCategoryTable.id,
                agileTenderId: schema.tenderKeywordCategoryTable.agileTenderId,
                normalTenderId: schema.tenderKeywordCategoryTable.normalTenderId,
                codeMercadoPublico: schema.tenderKeywordCategoryTable.codeMercadoPublico,
                itemDescription: schema.tenderKeywordCategoryTable.itemDescription,
            })
            .from(schema.tenderKeywordCategoryTable)
            .where(and(
                eq(schema.tenderKeywordCategoryTable.normalTenderId, normalTenderId),
                eq(schema.tenderKeywordCategoryTable.active, true)
            )).then((rows) => {
                return this.mapRowToTenderKeywordCategory(rows);
            });
    }
    async findAllByAgileTenderId(agileTenderId: number): Promise<TenderKeywordCategory[]> {
        this.LOGGER.log(`findByAgileTenderId agileTenderId: ${agileTenderId}`);
        return await this.db.conn
            .select({
                id: schema.tenderKeywordCategoryTable.id,
                agileTenderId: schema.tenderKeywordCategoryTable.agileTenderId,
                normalTenderId: schema.tenderKeywordCategoryTable.normalTenderId,
                codeMercadoPublico: schema.tenderKeywordCategoryTable.codeMercadoPublico,
                itemDescription: schema.tenderKeywordCategoryTable.itemDescription,
            })
            .from(schema.tenderKeywordCategoryTable)
            .where(and(
                eq(schema.tenderKeywordCategoryTable.agileTenderId, agileTenderId),
                eq(schema.tenderKeywordCategoryTable.active, true)
            )).then((rows) => {
                return this.mapRowToTenderKeywordCategory(rows);
            });
    }
    async upsert(insertData: InsertTenderKeywordCategory): Promise<number> {
        this.LOGGER.log(`upsert insertData: ${JSON.stringify(insertData)}`);
        const validatedTender = schema.tenderKeywordCategoryTableInsertSchema.parse({
            agileTenderId: insertData.agileTenderId,
            normalTenderId: insertData.normalTenderId,
            codeMercadoPublico: insertData.codeMercadoPublico,
            itemDescription: insertData.itemDescription,
        });
    
        const insertedTenderId = await this.db.conn
            .insert(schema.tenderKeywordCategoryTable)
            .values({
                agileTenderId: validatedTender.agileTenderId,
                normalTenderId: validatedTender.normalTenderId,
                codeMercadoPublico: validatedTender.codeMercadoPublico,
                itemDescription: validatedTender.itemDescription,
            })
            .onDuplicateKeyUpdate({
            set: {
                agileTenderId: validatedTender.agileTenderId,
                normalTenderId: validatedTender.normalTenderId,
                codeMercadoPublico: validatedTender.codeMercadoPublico,
                itemDescription: validatedTender.itemDescription,
                updatedAt: TenderUtil.getCurrentSystemDate(),
            },
            })
            .$returningId()
            .then((rows) => {
            return rows[0].id;
            });
    
        return insertedTenderId;
    }
    private mapRowToTenderKeywordCategory(rows: any): TenderKeywordCategory[] {
        return rows.map((row: any) => new TenderKeywordCategory(
            row.id,
            row.codeMercadoPublico,
            row.itemDescription,
            row.agileTenderId,
            row.normalTenderId
        ));
    }
}
