import { Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { and, count, desc, eq, inArray, isNull, sql } from 'drizzle-orm';
import { PrimeLogger, schema } from 'src/framework';
import { TenderRepository, Licitacion, AgileTenderFullInfoTO } from 'src/licitaapp';
import { GeneralDataInfoTO, InsertTender, Metadata, PaginationNormalTenderTO, RequestTender, Tender, TenderHistoryTO, TenderTO } from 'src/licitaapp/domain';
import { AgileTender, InsertAgileTender, MinimalToEditAgileTender, UpdateMinimalToEditAgileTender } from 'src/licitaapp/domain/entities/agile-tender.entity';
import { TenderRow } from 'src/licitaapp/domain/type/tender.types';
import { TenderUtil } from 'src/licitaapp/domain/util';
import { DBConfigService } from 'src/framework/infrastructure/drizzle/drizzle.provider';
import { PaginationAgileTenderTO } from 'src/licitaapp/domain/dto/pagination-agile-tender.to';


@Injectable()
export class TenderRepositoryImpl implements TenderRepository {
  private readonly LOGGER = new PrimeLogger(TenderRepositoryImpl.name);
  private readonly limitGeoTendersDashboard: number;
  constructor(private readonly db: DBConfigService,
    private readonly configService: ConfigService,
  ) {
    this.limitGeoTendersDashboard = this.configService.get<number>(
      'LIMIT_GEO_TENDERS_DASHBOARD',
      5
    );
  }
  async createGeneralDataQuery(): Promise<GeneralDataInfoTO> {
    const outPut = new GeneralDataInfoTO();
    await (this.db.conn).execute(
      sql.raw(`select count(id) as amount from company where active=1`)            
    ).then((rows:any) => {
      const items = rows[0];
      outPut.companyNumber = Number(items[0].amount ? items[0].amount : 0);
    });

    await (this.db.conn).execute(
      sql.raw(`select count(id) as amount from user where active=1`)            
    ).then((rows:any) => {
      const items = rows[0];
      outPut.usersNumber = Number(items[0].amount ? items[0].amount : 0);
    });

    await (this.db.conn).execute(
      sql.raw(`select count(id) as amount from tender_agile where active=1`)            
    ).then((rows:any) => {
      const items = rows[0];
      outPut.agileTenderNumber = Number(items[0].amount ? items[0].amount : 0);
    });

    await (this.db.conn).execute(
      sql.raw(`select count(id) as amount from tender where active=1`)            
    ).then((rows:any) => {
      const items = rows[0];
      outPut.normalTenderNumber = Number(items[0].amount ? items[0].amount : 0);
    });

    await (this.db.conn).execute(
      sql.raw(`SELECT COUNT(id) AS amount FROM user WHERE last_access >= CURDATE() AND last_access < CURDATE() + INTERVAL 1 DAY`)
    ).then((rows:any) => {
      const items = rows[0];
      outPut.activeNumber = Number(items[0].amount ? items[0].amount : 0);
    });

    await (this.db.conn).execute(
      sql.raw(`SELECT name, last_name FROM user WHERE active=1 ORDER BY last_access DESC LIMIT 1`)
    ).then((rows:any) => {
      const items = rows[0];
      outPut.fullNameLastAccess = items[0].name ? items[0].name : '' + ' ' + (items[0].last_name ? items[0].last_name : '');
    });

    await (this.db.conn).execute(
      sql.raw(`SELECT name, last_name FROM user WHERE active=1 ORDER BY created_at DESC LIMIT 1`)
    ).then((rows:any) => {
      const items = rows[0];
      outPut.fullNameLastRegister = items[0].name ? items[0].name : '' + ' ' + (items[0].last_name ? items[0].last_name : '');
    });

    return outPut;
  }
  async getNormalTenderAdminPaginated(page: number, pageSize: number, order: string, filter: string): Promise<PaginationNormalTenderTO[]> {
    this.LOGGER.log(`getAgileTenderAdminPaginated page ${page} pageSize ${pageSize} order ${order} filter ${filter}`);
    const offset = (page - 1) * pageSize;
    const outPutList : PaginationNormalTenderTO[] = [];
    let query = `SELECT id, code, name, description, subdivision_id, close_date FROM tender`;

    if (filter && filter.trim() !== '') {
      query += ` WHERE name like('%${filter}%') or name like('%${filter.toUpperCase()}%')`;
      query += ` AND description like('%${filter}%') or description like('%${filter.toUpperCase()}%')`;
    }

    query += ` ORDER BY ${order}
           LIMIT ${pageSize} OFFSET ${offset};`;

    await (this.db.conn).execute(
      sql.raw(query)
    ).then((rows:any) => {
      const items = rows[0];
      items.forEach((item: any) => {
      const out = new PaginationNormalTenderTO();
      out.id = item.id,
      out.code = item.code;
      out.name = item.name;
      out.description = item.description;
      out.subdivisionId = item.subdivision_id;
      out.closeDate = item.close_date;
      outPutList.push(out);
      });
    });

    return outPutList;
  }
  async getAgileTenderMinimalToEditById(agileTenderId: number): Promise<MinimalToEditAgileTender | undefined> {
    this.LOGGER.log(`getAgileTenderMinimalToEditById - agileTenderId: ${agileTenderId}`);
    return await this.db.conn
      .select({
        description: schema.agileTenderTable.description,
        subdivisionId: schema.agileTenderTable.subdivisionId,
      })
      .from(schema.agileTenderTable)
      .where(eq(schema.agileTenderTable.id, agileTenderId))
      .then((row: any) => {
        if (row.length === 0) {
          return undefined;
        }
        const output = new MinimalToEditAgileTender();
        output.description = row[0].description;
        output.subdivisionId = row[0].subdivisionId;
        return output;
      });
  }
  /*async updateSearchEntitiesAgileTender(dataAgiletender: AgileSearchEntityTO): Promise<void> {
    this.LOGGER.log(`updateSearchEntitiesAgileTender - agileTenderId: ${dataAgiletender.agileTenderId}, searchEntities: ${dataAgiletender.searchEntities}`);
    await this.db.conn
      .update(schema.agileTenderTable)
      .set({ searchEntities: dataAgiletender.searchEntities, updatedAt: TenderUtil.getCurrentSystemDate() })
      .where(eq(schema.agileTenderTable.id, dataAgiletender.agileTenderId))
      .execute();
  }*/

  async updateMinimalAgileTender(dataAgiletender: UpdateMinimalToEditAgileTender): Promise<void> {
    this.LOGGER.log(`updateMinimalAgileTender - agileTenderId: ${dataAgiletender}`);
    await this.db.conn
      .update(schema.agileTenderTable)
      .set({ description: dataAgiletender.description, subdivisionId: dataAgiletender.subdivisionId, updatedAt: TenderUtil.getCurrentSystemDate() })
      .where(eq(schema.agileTenderTable.id, dataAgiletender.id))
      .execute();
  }

  async getAgileTenderById(agileTenderId: number): Promise<AgileTender | null | undefined> {
    this.LOGGER.log(`getAgileTenderById - agileTenderId: ${agileTenderId}`);
    return await this.db.conn
      .select({
        id: schema.agileTenderTable.id,
        code: schema.agileTenderTable.code,
        name: schema.agileTenderTable.name,
        description: schema.agileTenderTable.description,
        details: schema.agileTenderTable.details,
        createdAt: schema.agileTenderTable.createdAt,
        updatedAt: schema.agileTenderTable.updatedAt,
        closeDate: schema.agileTenderTable.closeDate,
        subdivisionId: schema.agileTenderTable.subdivisionId,
        descriptionLocation: schema.agileTenderTable.descriptionLocation,
      })
      .from(schema.agileTenderTable)
      .where(eq(schema.agileTenderTable.id, agileTenderId))
      .then((row: any) => {
        if (row.length === 0) {
          return undefined;
        }
        return this.mapRowToAgileTender([row[0]])[0];
      });

  }
  async getAgileTenderAdminPaginated(page: number, pageSize: number, order: string, filter: string): Promise<PaginationAgileTenderTO[]> {
    this.LOGGER.log(`getAgileTenderAdminPaginated page ${page} pageSize ${pageSize} order ${order} filter ${filter}`);
    const offset = (page - 1) * pageSize;
    const outPutList : PaginationAgileTenderTO[] = [];
    let query = `SELECT id, code, name, description, close_date
           FROM tender_agile`;

    if (filter && filter.trim() !== '') {
      query += ` WHERE details like('%${filter}%') or details like('%${filter.toUpperCase()}%')`;
    }

    query += ` ORDER BY ${order}
           LIMIT ${pageSize} OFFSET ${offset};`;

    await (this.db.conn).execute(
      sql.raw(query)
    ).then((rows:any) => {
      const items = rows[0];
      items.forEach((item: any) => {
      const out = new PaginationAgileTenderTO();
      out.id = item.id,
      out.code = item.code;
      out.name = item.name;
      out.description = item.description;
      outPutList.push(out);
      });
    });

    return outPutList;
  }
  /*async updateMetadataAgileTender(agileTenderId: number, metadata: Metadata, searchEntities: string): Promise<void> {
    this.LOGGER.log(`updateMetadataAgileTender Agile Tender: ${agileTenderId}`);
    await this.db.conn
      .update(schema.agileTenderTable)
      .set({ metadata, updatedAt: TenderUtil.getCurrentSystemDate(), searchEntities })
      .where(eq(schema.agileTenderTable.id, agileTenderId))
      .execute();
    return Promise.resolve();
  }*/
  
  async findByColumnLike(column: string, value: string): Promise<Tender[]> {
    this.LOGGER.log(`findByColumnLike column: ${column} value: ${value}`);

    var listTender: Tender[] = [];
    let querySQL = '';
    if('description' === column){
      querySQL = `SELECT * FROM tender WHERE MATCH(description) AGAINST ('${value}' IN NATURAL LANGUAGE MODE)`;
    }else {
      //TODO: FEATURE: mejorar esta query con alto consumo
      querySQL = `SELECT * FROM tender WHERE LOWER(details) LIKE LOWER('%${value}%');`;
    }
    await this.db.conn.execute(
      querySQL
    ).then((rows:any) => {
      const items = rows[0];
      listTender = items.map((item: any) => {
        return this.mapRowToTender(item);
      });
    });
    return listTender;
  }
  erraseAgileOldTenders(agileTenderIds: number[]): Promise<void> {
    this.LOGGER.log(`erraseAgileOldTenders Agile Tender IDs: ${agileTenderIds}`);
    this.db.conn.delete(schema.agileTenderTable)
      .where(inArray(schema.agileTenderTable.id, agileTenderIds)).execute();
    return Promise.resolve();
  }
  async getAgileTenderPaginated(page: number, pageSize: number, typeSearch: string, subdivisionIds: number[], companyId: number, userId: number): Promise<AgileTenderFullInfoTO[]> {
    this.LOGGER.log(`pagination agile page ${page} pageSize ${pageSize} typeSearch ${typeSearch} subdivisionIds ${subdivisionIds} companyId ${companyId}`);
    const offset = (page - 1) * pageSize;


    let whereSentence = and(
      eq(schema.agileTenderTable.active, true),
    );

    if(typeSearch === 'keyword'){
      whereSentence = and(whereSentence, eq(schema.userCompanyAgileTenderTable.userId, userId), eq(schema.userCompanyAgileTenderTable.companyId, companyId));
      return await this.db.conn.
        select({
          details: schema.agileTenderTable.details,
          createdAt: schema.agileTenderTable.createdAt,
          updatedAt: schema.agileTenderTable.updatedAt,
          tenderAgileId: schema.agileTenderTable.id,
          closeDate: schema.agileTenderTable.closeDate,
        })
        .from(schema.userCompanyAgileTenderTable)
        .leftJoin(schema.agileTenderTable, eq(schema.userCompanyAgileTenderTable.agileTenderId, schema.agileTenderTable.id))
        .where(whereSentence)
        .orderBy(desc(schema.agileTenderTable.closeDate))
        .offset(offset)
        .limit(Number(pageSize))
        .then((rows) => {
            if (rows.length === 0) {
            return [];
            }
            return rows.map(
            (row) =>{
              const out = row.details as AgileTenderFullInfoTO;
              out.general.lastUpdatedAt = row.updatedAt? row.updatedAt : row.createdAt;
              out.general.tenderAgileId = row.tenderAgileId;
              out.general.closeDate = row.closeDate;
              return out;
            }
            );
        });

    }else if(typeSearch === 'subdivision' && subdivisionIds.length > 0){
      whereSentence = and(whereSentence, inArray(schema.agileTenderTable.subdivisionId, subdivisionIds));
    }else if(typeSearch === 'uncategorized'){
      whereSentence = and(whereSentence, isNull(schema.agileTenderTable.subdivisionId));
    }

    return await this.db.conn.
        select({
          details: schema.agileTenderTable.details,
          createdAt: schema.agileTenderTable.createdAt,
          updatedAt: schema.agileTenderTable.updatedAt,
          tenderAgileId: schema.agileTenderTable.id,
          closeDate: schema.agileTenderTable.closeDate,
        })
        .from(schema.agileTenderTable)
        .where(whereSentence)
        .orderBy(desc(schema.agileTenderTable.closeDate))
        .offset(offset)
        .limit(Number(pageSize))
        .then((rows) => {
            if (rows.length === 0) {
            return [];
            }
            return rows.map(
            (row) =>{
              const out = row.details as AgileTenderFullInfoTO;
              out.general.lastUpdatedAt = row.updatedAt? row.updatedAt : row.createdAt;
              out.general.tenderAgileId = row.tenderAgileId;
              out.general.closeDate = row.closeDate;
              return out;
            }
            );
        });
  }
  async getTotalAmountAgileTenders(userId: number, companyId: number, subdivisionIds: number[]): Promise<{ favoriteMoney:number, totalMoney: number; totalSubdivision: number; agileKeywordAmount:number }> {
    this.LOGGER.log(`getTotalAmountAgileTenders userId: ${userId}, companyId: ${companyId}, subdivisionIds: ${subdivisionIds}`);
    const output = {
      totalMoney: 0,
      totalSubdivision: 0,
      agileKeywordAmount: 0,
      favoriteMoney: 0,
    };

    await this.db.conn.execute(
      `SELECT count(id) AS total_amount_clp FROM tender_agile WHERE active = 1`
    ).then((rows:any) => {
      const items = rows[0];
      output.totalMoney = Number(items[0].total_amount_clp ? items[0].total_amount_clp : 0);
    });

    await this.db.conn.execute(
      `SELECT subdivision_id, SUM(amount_clp) AS total_amount_clp FROM tender_agile WHERE deleted_at IS NULL
  AND active = 1 AND subdivision_id IN (${subdivisionIds.join(',')}) GROUP BY subdivision_id`
    ).then((rows:any) => {
      const items = rows[0];
      let amountPerSubdivision = 0;
      items.forEach((item: any) => {
        amountPerSubdivision += Number( item.total_amount_clp ? item.total_amount_clp : 0);
      });
      output.totalSubdivision = amountPerSubdivision;
    })

    await this.db.conn.execute(
      `SELECT SUM(ta.amount_clp) AS total_amount_clp FROM 
            user_company_agile_tender tkc LEFT JOIN tender_agile ta 
        ON tkc.agile_tender_id = ta.id WHERE tkc.user_id=${userId} and tkc.is_favorite=1 and 
            tkc.company_id=${companyId} AND tkc.active = 1 AND ta.active = 1 GROUP BY ta.code`).then((rows:any) => {
      const items = rows[0];
      let favoriteMoney = 0;
      items.forEach((item: any) => {
        favoriteMoney += Number( item.total_amount_clp ? item.total_amount_clp : 0);
      });
      output.favoriteMoney = favoriteMoney;
    })

    await this.db.conn.execute(
      `SELECT SUM(ta.amount_clp) AS total_amount_clp FROM 
          user_company_agile_tender tkc LEFT JOIN tender_agile ta 
      ON tkc.agile_tender_id = ta.id WHERE tkc.user_id=${userId} and 
          tkc.company_id=${companyId} AND tkc.active = 1 AND ta.active = 1 GROUP BY ta.code`
    ).then((rows:any) => {
      const items = rows[0];
      let agileKeywordAmount = 0;
      items.forEach((item: any) => {
        agileKeywordAmount += Number( item.total_amount_clp ? item.total_amount_clp : 0);
      });
      output.agileKeywordAmount = agileKeywordAmount;
    })

    return output;
  }
  
  async findInfohistoryTender(tenderId: number[]): Promise<TenderHistoryTO[]> {
    this.LOGGER.log(`findInfohistoryTender Tender IDs: ${tenderId.filter(id => id != 0)}`);
    if(tenderId.length === 0){
      return [];
    }
    const selectFields = {
      tenderId: schema.tenderTable.id,
      code: schema.tenderTable.code,
      name: schema.tenderTable.name,
      description: schema.tenderTable.description,
      details: schema.tenderTable.details,
      createdAt: schema.tenderTable.createdAt,
      updatedAt: schema.tenderTable.updatedAt,
    };

    return await (this.db.conn)
      .select(selectFields)
      .from(schema.tenderTable)
      .where(inArray(schema.tenderTable.id, tenderId.filter(id => id != 0)))
      .then((rows: any[]) => {
        if (rows.length === 0) {
          return [];
        }
        return rows.map((row) => {
          return {
            tenderId: row.tenderId,
            code: row.code,
            name: row.name,
            description: row.description,
            details: row.details,
            createdAt: row.createdAt,
            updatedAt: row.updatedAt,
          } as TenderHistoryTO;
        });
      });
  }
  async erraseByListId(tenderIds: number[]): Promise<void> {
    this.LOGGER.log(`erraseByListId Tender IDs: ${tenderIds}`);
    await this.db.conn
      .delete(schema.tenderTable)
      .where(inArray(schema.tenderTable.id, tenderIds)).execute();
  }
    async findByCode(code: string): Promise<Tender | undefined> {
    return await this.selectTenderQuery()
      .where(and(eq(schema.tenderTable.code, code), eq(schema.tenderTable.active, true)))
      .then((rows: TenderRow[]) => {
        if (rows.length === 0) {
          return undefined;
        }

        return this.mapRowToTender(rows[0]);
      });
  }

  async logicalRemove(tenderId: number): Promise<void> {
    this.LOGGER.log(`logicalRemove Tender: ${tenderId}`);
    await (this.db.conn)
      .update(schema.tenderTable)
      .set({ 
        active: false,
        deletedAt: TenderUtil.getCurrentSystemDate(),
      })
      .where(eq(schema.tenderTable.id, tenderId));
  }

  /*async getAllWithouthMetadata(tx?: TxType) {
    this.LOGGER.log(`getAllWithouthMetadata`);
    return await (tx || this.db.conn)
      .select({
        id: schema.tenderTable.id,
        description: schema.tenderTable.description,
        detail: schema.tenderTable.details,
      })
      .from(schema.tenderTable)
      .where(and(eq(schema.tenderTable.active, true), isNull(schema.tenderTable.metadata)))
      .then((rows) => {
        return rows.map((row) => ({
          id: row.id,
          description: row.description,
          detail: row.detail
        }));
      }
    );
  }*/

  /*async updateMetadata(tenderId: number, metadata: Metadata){
    this.LOGGER.log(`updateMetadata Tender: ${tenderId}`);    
    await (this.db.conn)
      .update(schema.tenderTable)
      .set({ 
        updatedAt: TenderUtil.getCurrentSystemDate(), 
        metadata: metadata, 
      })
      .where(eq(schema.tenderTable.id, tenderId));
  }*/

  async findByDates(initDate: Date, endDate: Date ): Promise<RequestTender[]> {
    this.LOGGER.log(`findByDates - initDate: ${initDate} endDate: ${endDate}`);

    var listTender: RequestTender[] = [];
    await this.db.conn.execute(
      `SELECT id, code, name, description, metadata FROM tender WHERE created_at BETWEEN '${initDate.toISOString().slice(0, 19).replace("T", " ")}' AND '${endDate.toISOString().slice(0, 19).replace("T", " ")}' AND active = true AND metadata IS NOT NULL`
    ).then((rows:any) => {
      const items = rows[0];
      listTender = items.map((item: any) => {
        const newTender = new RequestTender();
        newTender.id = item.id;
        newTender.code = item.code;
        newTender.name = item.name;
        newTender.description = item.description;
        newTender.metadata = item.metadata;
        return newTender;
      });
    })
    .catch((err) => {
      this.LOGGER.error(`Error findByDates: ${err}`);
    });
    
    return listTender;
  }

  async findWithoutCloseDate(): Promise<Tender[]> {
    this.LOGGER.log(`findWithoutCloseDate`);
    var listTender: Tender[] = [];
    await this.db.conn.execute(
      `SELECT * FROM tender WHERE close_date IS NULL`
    ).then((rows:any) => {
      const items = rows[0];
      listTender = items.map((item: any) => {
        return this.mapRowToTender(item);
      });
    })
    .catch((err) => {
      this.LOGGER.error(`Error findWithoutCloseDate: ${err}`);
    });
    
    return listTender;
  }

  async findWithoutSubdivision(): Promise<Tender[]> {
    this.LOGGER.log(`findWithoutCity`);
    var listTender: Tender[] = [];
    await this.db.conn.execute(
      `SELECT * FROM tender WHERE subdivision_id IS NULL`
    ).then((rows:any) => {
      const items = rows[0];
      listTender = items.map((item: any) => {
        return this.mapRowToTender(item);
      });
    })
    .catch((err) => {
      this.LOGGER.error(`Error findWithoutCity: ${err}`);
    });
    
    return listTender;
  }

  async findByCloseDate(endDate: Date ): Promise<Tender[]> {
    this.LOGGER.log(`findByCloaseDate - endDate: ${endDate}`);

    var listTender: Tender[] = [];
    await this.db.conn.execute(
      `SELECT * FROM tender WHERE close_date <= '${endDate.toISOString().slice(0, 19).replace("T", " ")}'`
    ).then((rows:any) => {
      const items = rows[0];
      listTender = items.map((item: any) => {
      return this.mapRowToTender(item);
      });
    })
    .catch((err) => {
      this.LOGGER.error(`Error findByCloseDate: ${err}`);
    });
    
    return listTender;
  }

  async findByCloseDateAgileTender(endDate: Date ): Promise<number[]> {
    this.LOGGER.log(`findByCloseDateAgileTender - endDate: ${endDate}`);

    var listTender: number[] = [];
    await this.db.conn.execute(
      `SELECT id FROM agile_tender WHERE close_date <= '${endDate.toISOString().slice(0, 19).replace("T", " ")}' AND active = true`
    ).then((rows:any) => {
      const items = rows[0];
      listTender = items.map((item: any) => {
      return item as number;
      });
    })
    .catch((err) => {
      this.LOGGER.error(`Error findByCloseDate: ${err}`);
    });
    
    return listTender;
  }

  async findByDatesUpset(initDate: Date, endDate: Date ): Promise<Tender[]> {
    this.LOGGER.log(`findByDates - initDate: ${initDate.toISOString().slice(0, 19).replace("T", " ")} endDate: ${endDate.toISOString().slice(0, 19).replace("T", " ")}`);

    var listTender: Tender[] = [];
    await this.db.conn.execute(
      `SELECT * FROM tender WHERE created_at BETWEEN '${initDate.toISOString().slice(0, 19).replace("T", " ")}' AND '${endDate.toISOString().slice(0, 19).replace("T", " ")}' AND active = true OR 
      updated_at BETWEEN '${initDate.toISOString().slice(0, 19).replace("T", " ")}' AND '${endDate.toISOString().slice(0, 19).replace("T", " ")}' AND active = true`
    ).then((rows:any) => {
      const items = rows[0];
      listTender = items.map((item: any) => {
        return this.mapRowToTender(item);
      });
    })
    .catch((err) => {
      this.LOGGER.error(`Error findByDates: ${err}`);
    });
    
    return listTender;
  }

  async findById(id: number): Promise<Tender | undefined> {
    return await this.selectTenderQuery()
      .where(and(eq(schema.tenderTable.id, id), eq(schema.tenderTable.active, true)))
      .then((rows: TenderRow[]) => {
        if (rows.length === 0) {
          return undefined;
        }

        return this.mapRowToTender(rows[0]);
      });
  }

  /**
   * Si estan activas significa que aun estan en curso.
   * @param subdivisionId 
   * @returns 
   */
  async findBySubdivisionId(subdivisionId: number, withLimit: boolean = false): Promise<Tender[]> {
    try {
      const query = this.selectTenderQuery()
        .where(and(eq(schema.tenderTable.subdivisionId, subdivisionId), eq(schema.tenderTable.active, true)));

      if (withLimit) {
        query.limit(Number(this.limitGeoTendersDashboard));
      }
      query.orderBy(desc(schema.tenderTable.createdAt));
      const rows = await query;
      return rows.map((item: any) => this.mapRowToTender(item));
    } catch (err) {
      this.LOGGER.error(`Error findBySubdivisionId: ${err}`);
      return [];
    }
  }

  async upsert(tender: InsertTender): Promise<number> {
    const validatedTender = schema.tenderTableInsertSchema.parse({
      code: tender.code,
      name: tender.name,
      description: tender.description,
      details: tender.details,
      subdivisionId: tender.subdivisionId,
    });

    const insertedTenderId = await this.db.conn
      .insert(schema.tenderTable)
      .values({
        code: validatedTender.code,
        name: validatedTender.name,
        description: validatedTender.description,
        subdivisionId: validatedTender.subdivisionId,
        details: tender.details,
      })
      .onDuplicateKeyUpdate({
        set: {
          code: validatedTender.code,
          name: validatedTender.name,
          description: validatedTender.description,
          subdivisionId: validatedTender.subdivisionId,
          details: tender.details,
          updatedAt: TenderUtil.getCurrentSystemDate(),
        },
      })
      .$returningId()
      .then((rows) => {
        return rows[0].id;
      });

    return insertedTenderId;
  }

  async countActiveTenders(): Promise<number> {
    this.LOGGER.log(`countActiveTenders`);
      return await this.db.conn
        .select({count: count()})
        .from(schema.tenderTable)
        .where(
          and(
            eq(schema.tenderTable.active, true),
          ),
        )
        .then((rows) => {
          return rows[0].count;
        });
  }

  async getLogicalRemoveTenderIds(): Promise<TenderTO[]> {
    this.LOGGER.log(`getLogicalRemoveTenderIds`);
    return await this.db.conn
      .select({ id: schema.tenderTable.id, code: schema.tenderTable.code, details: schema.tenderTable.details, description: schema.tenderTable.description })
      .from(schema.tenderTable)
      .where(eq(schema.tenderTable.active, false))
      .then((rows) => {
        return rows.map((row) => new TenderTO(row.id, row.code, row.details, row.description));
      });
  }

  async updateField(tender: {
    id: number;
    active: boolean;
    detail?: Licitacion;
    subdivisionId?: number | null;
  }): Promise<void> {
    
    await this.db.conn
      .update(schema.tenderTable)
      .set({
        active: tender.active,
        subdivisionId: tender.subdivisionId,
        updatedAt: TenderUtil.getCurrentSystemDate(),
        details: tender.detail,
        deletedAt: tender.active ? null : TenderUtil.getCurrentSystemDate(),
      })
      .where(eq(schema.tenderTable.id, tender.id))
      .execute();
  }

  async updateCloseDate(tender: {
    id: number;
    closeDate: Date | null;
  }): Promise<void> {
    
    await this.db.conn
      .update(schema.tenderTable)
      .set({
        closeDate: tender.closeDate,
        updatedAt: TenderUtil.getCurrentSystemDate()
      })
      .where(eq(schema.tenderTable.id, tender.id))
      .execute();
  }

  async updateLogicalRemove(tender: {
    id: number;
    state: boolean;
  }): Promise<void> {
    
    if(!tender.state){
      await this.db.conn
      .update(schema.tenderTable)
      .set({
        active: tender.state,
        deletedAt: TenderUtil.getCurrentSystemDate()
      })
      .where(eq(schema.tenderTable.id, tender.id))
      .execute();
    }
  }

  async updateSubdivision(tender: {
    id: number;
    subdivisionId?: number | null;
  }): Promise<void> {
    
    await this.db.conn
      .update(schema.tenderTable)
      .set({
        subdivisionId: tender.subdivisionId,
        updatedAt: TenderUtil.getCurrentSystemDate()
      })
      .where(eq(schema.tenderTable.id, tender.id))
      .execute();
  }
  
  private selectTenderQuery() {
    return this.db.conn
      .select({
        id: schema.tenderTable.id,
        code: schema.tenderTable.code,
        name: schema.tenderTable.name,
        description: schema.tenderTable.description,
        details: schema.tenderTable.details,
        createdAt: schema.tenderTable.createdAt,
        updatedAt: schema.tenderTable.updatedAt,
        closeDate: schema.tenderTable.closeDate,
      })
      .from(schema.tenderTable)
      .$dynamic();
  }
  
  private mapRowToTender(row: TenderRow): Tender {
    return new Tender(
      row.id,
      row.code,
      row.name,
      row.description,
      row.details,
      row.createdAt,
      row.updatedAt,
      row.closeDate,
      false,
      row.subdivisionId ? row.subdivisionId : undefined,
    );
  }

  private mapRowToAgileTender(rows: any): AgileTender[] {
    return rows.map((row: any) => new AgileTender(
      row.id,
      row.code,
      row.name,
      row.description,
      row.details,
      row.createdAt,
      row.closeDate,
      row.isFavorite ? row.isFavorite : false,
      row.descriptionLocation ? row.descriptionLocation : undefined,
      row.amountCLP ? row.amountCLP : undefined,
      row.subdivisionId ? row.subdivisionId : undefined,
      row.updatedAt ? row.updatedAt : undefined,
    ));
  }

  async upsertAgileTender(tender: InsertAgileTender): Promise<number> {
    const validatedTender = schema.agileTenderTableInsertSchema.parse({
      code: tender.code,
      name: tender.name,
      closeDate: tender.closeDate,
      description: tender.description,
      descriptionLocation: tender.descriptionLocation,
      details: tender.details,
      subdivisionId: tender.subdivisionId,
      amountCLP: tender.amountCLP,
    });

    const insertedTenderId = await this.db.conn
      .insert(schema.agileTenderTable)
      .values({
        code: validatedTender.code,
        name: validatedTender.name,
        closeDate: validatedTender.closeDate,
        description: validatedTender.description,
        descriptionLocation: validatedTender.descriptionLocation,
        details: tender.details,
        subdivisionId: validatedTender.subdivisionId,
        amountCLP: validatedTender.amountCLP,
      })
      .onDuplicateKeyUpdate({
        set: {
          code: validatedTender.code,
          name: validatedTender.name,
          closeDate: validatedTender.closeDate,
          description: validatedTender.description,
          descriptionLocation: validatedTender.descriptionLocation,
          details: tender.details,
          updatedAt: TenderUtil.getCurrentSystemDate(),
          subdivisionId: validatedTender.subdivisionId,
          amountCLP: validatedTender.amountCLP,
        },
      })
      .$returningId();

    return insertedTenderId[0].id;
  }

  async getAgileTenderWithouthMetadata(): Promise<AgileTender[]> {
    return await this.db.conn
      .select({
        id: schema.agileTenderTable.id,
        code: schema.agileTenderTable.code,
        name: schema.agileTenderTable.name,
        description: schema.agileTenderTable.description,
        details: schema.agileTenderTable.details,
        createdAt: schema.agileTenderTable.createdAt,
        updatedAt: schema.agileTenderTable.updatedAt,
        closeDate: schema.agileTenderTable.closeDate,
        subdivisionId: schema.agileTenderTable.subdivisionId,
        descriptionLocation: schema.agileTenderTable.descriptionLocation,
      })
      .from(schema.agileTenderTable)
      //.where(and(isNull(schema.agileTenderTable.metadata), eq(schema.agileTenderTable.active, true)))
      .where(and(eq(schema.agileTenderTable.active, true)))
      // el limite de la api es 500
      .limit(500)
      .then((rows: any[]) => {
        return this.mapRowToAgileTender(rows);
      });
  }

}
