import { Injectable } from '@nestjs/common';
import { and, asc, count, eq, inArray } from 'drizzle-orm';
import { PrimeLogger, schema } from 'src/framework';
import { DBConfigService } from 'src/framework/infrastructure/drizzle/drizzle.provider';
import { AgileTenderFullInfoTO } from 'src/licitaapp/application';
import { UserCompanyAgileTenderRepository } from 'src/licitaapp/application/repository/user-company-agile-tender-repository/user-company-agile-tender-repository.interface';
import { AgileTender } from 'src/licitaapp/domain';
import { TenderUtil } from 'src/licitaapp/domain/util';

@Injectable()
export class UserCompanyAgileTenderRepositoryImpl
  implements UserCompanyAgileTenderRepository
{
  private readonly LOGGER = new PrimeLogger(
    UserCompanyAgileTenderRepositoryImpl.name,
  );
  constructor(private readonly db: DBConfigService) {}
  async findTenderByUserCompanyId(userId: number, companyId: number): Promise<AgileTender[]> {
    this.LOGGER.log(
      `findTenderByUserCompanyId userId: ${userId} companyId: ${companyId}`
    );
    return await this.db.conn
      .select(
        {
          id: schema.agileTenderTable.id,
          code: schema.agileTenderTable.code,
          name: schema.agileTenderTable.name,
          description: schema.agileTenderTable.description,
          details: schema.agileTenderTable.details,
          createdAt: schema.agileTenderTable.createdAt,
          updatedAt: schema.agileTenderTable.updatedAt,
          closeDate: schema.agileTenderTable.closeDate,
          subdivisionId: schema.agileTenderTable.subdivisionId,
          descriptionLocation: schema.agileTenderTable.descriptionLocation,
        }
      )
      .from(schema.agileTenderTable)
      .innerJoin(
        schema.userCompanyAgileTenderTable,
        eq(schema.userCompanyAgileTenderTable.agileTenderId, schema.agileTenderTable.id),
      )
      .where(
        and(
          eq(schema.userCompanyAgileTenderTable.userId, userId),
          eq(schema.userCompanyAgileTenderTable.companyId, companyId),
          eq(schema.userCompanyAgileTenderTable.active, true),
          eq(schema.agileTenderTable.active, true),
        ),
      )
      .then((rows) => {
        return this.mapRowToAgileTender(rows);
      });
  }

  private mapRowToAgileTender(rows: any): AgileTender[] {
    return rows.map((row: any) => new AgileTender(
      row.id,
      row.code,
      row.name,
      row.description,
      row.details,
      row.createdAt,
      row.closeDate,
      row.isFavorite ? row.isFavorite : false,
      row.descriptionLocation ? row.descriptionLocation : undefined,
      row.amountCLP ? row.amountCLP : undefined,
      row.subdivisionId ? row.subdivisionId : undefined,
      row.updatedAt ? row.updatedAt : undefined,
    ));
  }

  getFavoriteList(userId: number): Promise<number[]> {
    this.LOGGER.log(`getFavoriteList - userId: ${userId}`);
    return this.db.conn
      .select({
        agileTenderId: schema.userCompanyAgileTenderTable.agileTenderId
      })
      .from(schema.userCompanyAgileTenderTable)
      .where(and(eq(schema.userCompanyAgileTenderTable.userId, userId), eq(schema.userCompanyAgileTenderTable.isFavorite, true)))
      .execute()
      .then((rows) => rows.map((row) => row.agileTenderId));
  }
  async getPaginatedAgileTenders(
    userId: number,
    companyId: number,
    page: number,
    pageSize: number,
    isFavorite: boolean,
  ): Promise<AgileTenderFullInfoTO[]> {
    this.LOGGER.log(
      `getPaginatedAgileTenders - userId: ${userId}, companyId: ${companyId}, page: ${page}, pageSize: ${pageSize}, isFavorite: ${isFavorite}`,
    );
    const offset = (page - 1) * pageSize;
    const selectFields = {
        data: schema.agileTenderTable.details,
        createdAt: schema.agileTenderTable.createdAt,
        updatedAt: schema.agileTenderTable.updatedAt,
        tenderAgileId: schema.agileTenderTable.id,
        closeDate: schema.agileTenderTable.closeDate,
    }

    let whereSentence = and(
        eq(schema.userCompanyAgileTenderTable.userId, userId),
        eq(schema.userCompanyAgileTenderTable.companyId, companyId),
        eq(schema.userCompanyAgileTenderTable.isFavorite, isFavorite),
        eq(schema.agileTenderTable.active, true)
    );

    const query = this.db.conn
      .select(selectFields)
      .from(schema.agileTenderTable)
      .where(whereSentence)
      .limit(Number(pageSize))
      .offset(offset);

    query.leftJoin(
        schema.userCompanyAgileTenderTable,
        eq(schema.userCompanyAgileTenderTable.agileTenderId, schema.agileTenderTable.id),
    );
    query.orderBy(asc(schema.agileTenderTable.closeDate));

    const rows = await query;

    if (rows.length === 0) {
      return [];
    }
    return rows.map(
      (row) =>{ 
        const out = row.data as AgileTenderFullInfoTO;
        out.general.lastUpdatedAt = row.updatedAt? row.updatedAt : row.createdAt;
        out.general.tenderAgileId = row.tenderAgileId;
        out.general.closeDate = row.closeDate;
        return out;
      }
       
    );
  }
  async saveAll(
    userId: number,
    companyId: number,
    tenderIds: number[],
    isFavorite: boolean,
  ): Promise<boolean> {
    this.LOGGER.log(
      `saveAll - userId: ${userId}, companyId: ${companyId}, tenderIds: ${tenderIds} isFavorite: ${isFavorite}`,
    );
    await this.db.conn.transaction(async (tx) => {
      for (const agileTenderId of tenderIds) {
        await tx
          .insert(schema.userCompanyAgileTenderTable)
          .values({ userId, companyId, agileTenderId, isFavorite })
          .onDuplicateKeyUpdate({
            set: {
              userId,
              companyId,
              agileTenderId,
              isFavorite,
              updatedAt: TenderUtil.getCurrentSystemDate(),
            },
          })
          .execute();
      }
    });
    return true;
  }

  async erraseUserCompanyAgileTender(tenderAgileIds: number[]): Promise<void> {
    this.LOGGER.log(
      `erraseUserCompanyAgileTender - tenderAgileIds: ${tenderAgileIds}`,
    );
    await this.db.conn
      .delete(schema.userCompanyAgileTenderTable)
      .where(
        inArray(
          schema.userCompanyAgileTenderTable.agileTenderId,
          tenderAgileIds,
        ),
      )
      .execute();
  }

   async countActiveTendersCompany(companyId: number): Promise<number> {
      return await this.db.conn
        .select({ count: count() })
        .from(schema.userCompanyAgileTenderTable)
        .where(
          and(
            eq(schema.userCompanyAgileTenderTable.companyId, companyId),
            eq(schema.userCompanyAgileTenderTable.active, true),
          ),
        )
        .then((rows) => {
          return rows[0].count;
        });
    }
}
