import { forwardRef, Inject, Injectable } from '@nestjs/common';
import { and, count, eq } from 'drizzle-orm';
import { PrimeLogger, schema, TxType } from 'src/framework';
import { DBConfigService } from 'src/framework/infrastructure/drizzle/drizzle.provider';
import { UserCompanyRepository } from 'src/licitaapp/application/repository/user-company-repository/user-company-repository.interface';
import { Company } from 'src/licitaapp/domain';
import { UserMinimalTO } from 'src/licitaapp/domain/dto/user.minimal.to';

@Injectable()
export class UserCompanyRepositoryImpl implements UserCompanyRepository {
  private readonly LOGGER = new PrimeLogger(UserCompanyRepositoryImpl.name);
  constructor(private readonly db: DBConfigService) {}
  async countActiveUser(): Promise<number> {
    return await this.db.conn
      .select({count: count()})
      .from(schema.userTable)
      .where(eq(schema.userTable.active, true))
      .then((rows) => {
        return rows[0].count;
      });
  }
  async countUserByCompany(companyId: number): Promise<number> {
    this.LOGGER.log(`Counting users by company: ${companyId}`);
    return await this.db.conn
      .select({count: count()})
      .from(schema.userCompanyTable)
      .where(
        and(
          eq(schema.userCompanyTable.companyId, companyId),
        ),
      )
      .then((rows) => {
        return rows[0].count;
      });
  }
  async getInfoUserCompany(companyId: number): Promise<UserMinimalTO[]> {
    this.LOGGER.log(`getInfoUserCompany companyId ${companyId}`);
    return await this.db.conn.
      select({
        userId: schema.userTable.id, 
        active: schema.userTable.active, 
        email: schema.userTable.email,
        cellPhone: schema.userTable.cellPhone,
        name: schema.userTable.name,
        lastName: schema.userTable.lastName,
      })
      .from(schema.companyTable)
      .innerJoin(
        schema.userCompanyTable,
        eq(schema.userCompanyTable.companyId, schema.companyTable.id),
      )
      .innerJoin(
        schema.userTable,
        eq(schema.userCompanyTable.userId, schema.userTable.id),
      )
      .where(and(eq(schema.userCompanyTable.companyId, companyId)))
      .then((rows) => {
        if (rows.length === 0) {
          return [];
        }
        return rows.map(
          (row) =>{
            const fullName = row.name + ' ' + (row.lastName?row.lastName : '');
            return new UserMinimalTO(
              row.userId,
              fullName,
              row.email,
              row.cellPhone? row.cellPhone : '',
              row.active
            );
          }
        );
    });
  }
  async getUserIdsWithActiveCompany(): Promise<number[]> {
    this.LOGGER.log(`getUserIdsWithActiveCompany`);
    return await this.db.conn
      .select({
        userId: schema.userCompanyTable.userId,
      })
      .from(schema.userCompanyTable)
      .innerJoin(
        schema.companyTable,
        eq(schema.userCompanyTable.companyId, schema.companyTable.id),
      )
      .where(eq(schema.companyTable.active, true))
      .groupBy(schema.userCompanyTable.userId)
      .then((rows) => {
        if (rows.length === 0) {
          return [];
        }

        return rows.map(
          (row) => row.userId,
        );
      });
  }
  async findCompaniesByUserId(userId: number): Promise<Company[]> {
    this.LOGGER.log(`Finding companies by user: ${userId}`);
    return await this.db.conn
      .select({
        id: schema.companyTable.id,
        dni: schema.companyTable.dni,
        socialReason: schema.companyTable.socialReason,
        updatedAt: schema.companyTable.updatedAt,
      })
      .from(schema.userCompanyTable)
      .innerJoin(
        schema.companyTable,
        eq(schema.userCompanyTable.companyId, schema.companyTable.id),
      )
      .where(
        and(
          eq(schema.userCompanyTable.userId, userId),
          eq(schema.companyTable.active, true),
        ),
      )
      .then((rows) => {
        if (rows.length === 0) {
          return [];
        }

        return rows.map(
          (row) => new Company(row.id, row.dni, row.socialReason),
        );
      });
  }
  async save(userId: number, companyId: number, tx?: TxType): Promise<boolean> {
    this.LOGGER.log(`Saving user company: ${userId} - ${companyId}`);
    try {
      await (tx || this.db.conn)
        .insert(schema.userCompanyTable)
        .values({ userId, companyId });
      return true;
    } catch (error) {
      throw new Error(error);
    }
  }
  async deleteByUserCompany(
    userId: number,
    companyId: number,
    tx?: TxType,
  ): Promise<void> {
    this.LOGGER.log(`Deleting user company: ${userId} - ${companyId}`);
    await (tx || this.db.conn)
      .delete(schema.userCompanyTable)
      .where(
        and(
          eq(schema.userCompanyTable.userId, userId),
          eq(schema.userCompanyTable.companyId, companyId),
        ),
      );
  }
  async paginationByUser(
    userId: number,
    page: number,
    pageSize: number,
  ): Promise<Company[]> {
    this.LOGGER.log(
      `paginationByUser user ${userId} page ${page} pageSize ${pageSize}`,
    );
    const offset = (page - 1) * pageSize;

    return await this.db.conn
      .select({
        field1: schema.companyTable.id,
        field2: schema.companyTable.dni,
        field3: schema.companyTable.socialReason,
        field4: schema.companyTable.createdAt,
        field5: schema.companyTable.updatedAt,
      })
      .from(schema.companyTable)
      .innerJoin(
        schema.userCompanyTable,
        eq(schema.userCompanyTable.companyId, schema.companyTable.id),
      )
      .where(
        and(
          eq(schema.userCompanyTable.userId, userId),
          eq(schema.companyTable.active, true),
        ),
      )
      .orderBy(schema.companyTable.dni)
      .offset(offset)
      .limit(Number(pageSize))
      .then((rows) => {
        if (rows.length === 0) {
          return [];
        }
        return rows.map((row) => {
          let company = new Company(row.field1, row.field2, row.field3);
          company.updatedAt = row.field5 ? row.field5 : row.field4;
          return company;
        });
      });
  }
}
