import { Inject, Injectable } from '@nestjs/common';
import { CacheManagerEnum, FwkCacheService, FwkCacheServiceImpl, PrimeLogger } from 'src/framework';
import { Subdivision } from 'src/framework/domain/entities/subdivision.entity';
import { DBConfigService } from 'src/framework/infrastructure/drizzle/drizzle.provider';
import {
  UserCompanyService,
  CompanyRepository,
  CompanyService,
  KeywordService,
} from 'src/licitaapp';
import { CompanyKeywordService } from 'src/licitaapp/application/service/company-keyword-service/company-keyword-service.interface';
import { TenderKeywordCategoryService } from 'src/licitaapp/application/service/tender-keyword-category-service/tender-keyword-category-service.interface';
import { UserCompanyAgileTenderService } from 'src/licitaapp/application/service/user-company-agile-tender-service/user-company-agile-tender-service.interface';
import { UserCompanyTenderService } from 'src/licitaapp/application/service/user-company-tender-service/user-company-tender-service.interface';
import { Company, CompanyFullTO, CompanyRegisterTO, Keyword, MinimalCompany } from 'src/licitaapp/domain';
import { PaginationKeywordTO } from 'src/licitaapp/domain/dto/company.user.to';
import { KeywordCompanyTO } from 'src/licitaapp/domain/dto/keyword-company.to';
import { UserPaginationTO } from 'src/licitaapp/domain/dto/user.pagination.to';

@Injectable()
export class CompanyServiceImpl implements CompanyService {
  private readonly LOGGER = new PrimeLogger(CompanyServiceImpl.name);
  constructor(
    @Inject('CompanyRepository')
    private readonly companyRepository: CompanyRepository,
    @Inject('UserCompanyService')
    private readonly userCompanyService: UserCompanyService,
    @Inject('TenderKeywordCategoryService') private readonly tenderKeywordCategoryService: TenderKeywordCategoryService,
    @Inject('KeywordService') private readonly keywordService: KeywordService,
    @Inject('UserCompanyTenderService') private readonly userCompanyTenderService: UserCompanyTenderService,
    @Inject('UserCompanyAgileTenderService') private readonly userCompanyAgileTenderService: UserCompanyAgileTenderService,
    @Inject('CompanyKeywordService') private readonly companyKeywordService: CompanyKeywordService,
    private readonly db: DBConfigService,
    @Inject('FwkCacheService')
    private readonly fwkCacheService: FwkCacheService,
  ) {}
  /*//TODO: reparar async findCompaniesKeywords(keywordCompanyTO: KeywordCompanyTO): Promise<MinimalCompany[]> {
    this.LOGGER.log(`Finding companies by keywords: ${keywordCompanyTO}`);
    const keywordsIds = await this.keywordService.findKeywordsByNameList(keywordCompanyTO.keywordValue);
    return this.companyRepository.findCompaniesKeywords(keywordsIds);
  }*/
  async paginationByCompanyKeyword(page: number, pageSize: number, socialReason?: string): Promise<PaginationKeywordTO[]> {
    this.LOGGER.log(`Paginating companies by keyword, page: ${page}, pageSize: ${pageSize}, socialReason: ${socialReason}`);
    const data = await this.companyRepository.paginationByCompanyKeyword(page, pageSize, socialReason); 

    for(const company of data){
      company.listKeywords = await this.companyKeywordService.findByCompanyId(company.companyId);
      company.amountTenders = await this.userCompanyTenderService.countActiveTendersCompany(company.companyId);
      company.amountAgileTenders = await this.userCompanyAgileTenderService.countActiveTendersCompany(company.companyId);
    }

    return data;
  }
  getAllCompanyWithoutKeywords(): Promise<number[]> {
    this.LOGGER.log('Getting all company without keywords');
    return this.companyRepository.getAllCompanyWithoutKeywords();
  }
  async getCompanyIdsActive(): Promise<number[]> {
    this.LOGGER.log('Getting company ids active');
    const resultData = await this.fwkCacheService.getOrSet<number[]>(
      FwkCacheServiceImpl.createKey(CacheManagerEnum.IDS_ACTIVE_COMPANIES, ["active-companies"]),
        async () => {
            return this.companyRepository.getCompanyIdsActive()
      },
    );
    return resultData;
  }

  async deleteByUserCompany(userId: number, companyId: number): Promise<void> {
    this.LOGGER.log(
      `deleteByUserCompany userId ${userId} companyId ${companyId}`,
    );
    return await this.userCompanyService.deleteByUserCompany(userId, companyId);
  }
  async paginationByCompanyAdmin(
    page: number,
    pageSize: number,
  ): Promise<UserPaginationTO> {
    this.LOGGER.log(`Paginating companies by admin, page: ${page}, pageSize: ${pageSize}`);
    const output = new UserPaginationTO();
    output.amountUsers = await this.userCompanyService.countActiveUser();
    const companies = await this.companyRepository.paginationByCompanyAdmin(page, pageSize);
    for (const company of companies) {
      const userInfo = await this.userCompanyService.getInfoUserCompany(
        company.companyId,
      );
      company.listUsers = userInfo;
      company.amountTenders = await this.userCompanyTenderService.countActiveTendersCompany(company.companyId);
    }
    output.listUsers = companies;
    return output;
  }
  async getCompanyIdsToRecalculateTender(): Promise<number[]> {
    this.LOGGER.log('Getting company ids to recalculate tender');
    return await this.companyRepository.getCompanyIdsToRecalculateTender();
  }
  async getSubdivisionsByCompanyId(companyId: number): Promise<Subdivision[]> {
    this.LOGGER.log(`Getting subdivisions for companyId: ${companyId}`);
    /*const resultData = await this.fwkCacheService.getOrSet<Subdivision[]>(
      FwkCacheServiceImpl.createKey(CacheManagerEnum.SUBDIVISIONS_BY_COMPANY_ID, [companyId]),
        async () => {
            return await this.companyRepository.getSubdivisionsByCompanyId(companyId)
      },
    );*/
    return this.companyRepository.getSubdivisionsByCompanyId(companyId);
  }
  async updateCheckTenders(
    companyId: number,
    checkTender: boolean,
  ): Promise<boolean> {
    this.LOGGER.log(
      `updateCheckTenders companyId ${companyId} checkTender ${checkTender}`,
    );
    return await this.companyRepository.updateCheckTenders(
      companyId,
      checkTender,
    );
  }

  /*async getAllWithouthMetadata(): Promise<Keyword[]> {
    this.LOGGER.log('Getting keywords metadata');
    return await this.keywordService.getAllWithouthMetadata();
  }*/

  async updateCompany(
    companyId: number,
    userId: number,
    updateCompany: CompanyRegisterTO,
  ): Promise<CompanyFullTO | null> {
    this.LOGGER.log(`Updating company: ${updateCompany.company.id}`);

    await this.db.conn.transaction(async (tx) => {
      const companyUpdated = await this.companyRepository.update(
        companyId,
        updateCompany,
        tx,
      );

      if (!companyUpdated) {
        this.LOGGER.error('problem updating company');
        throw new Error('Problem updating company');
      }

      this.LOGGER.log('Deleted relations to subdivisions');
      await this.companyRepository.deletedCompanySubdivisions(companyId, tx);
      this.LOGGER.log('Set new relations to subdivisions');
      if ((updateCompany.listSubdivisions.length) > 0) {
        this.LOGGER.log(
          `Saving company subdivisions for company: ${companyId} `,
        );

        const newSubdivisions = updateCompany.listSubdivisions.filter(
          (subdivision) => subdivision.id !== 7777,
        );
        this.LOGGER.log(
          `Subdivisions: ${JSON.stringify(updateCompany.listSubdivisions)}`,
        );

        await this.companyRepository.saveCompanySubdivisions(
          companyId,
          newSubdivisions,
          tx,
        );
      }
      this.LOGGER.log(
        `Set new relations to keywords size: ${updateCompany.listKeywords.length}`,
      );
      if (updateCompany.listKeywords.length > 0) {
        await this.keywordService.saveAll(
          updateCompany.listKeywords.map((keyword) => ({
            ...keyword,
            companyId: +companyId,
            metadata: undefined,
          })),
          tx,
        );
        this.LOGGER.log(`Saving keywords for company: ${companyId}`);
      
        const listKeywordsDB = await this.keywordService.findByCodeCategoriaMercadoPublico(
          updateCompany.listKeywords.map((keyword) => keyword.codeCategoriaMercadoPublico),
          tx,
        );
        this.LOGGER.log(`Deleting old company keywords for company: ${companyId}`);
        await this.companyKeywordService.deleteByCompanyId(companyId, tx);
        this.LOGGER.log(`Saving new company keywords for company: ${companyId}`);
        await this.companyKeywordService.saveAll(
          companyId,
          listKeywordsDB.map((keyword) => keyword.id),
          tx,
        );
      }
    });
    
    await this.fwkCacheService.del();
    return this.findByIdWithFetch(companyId);
  }
  async createCompany(
    userId: number,
    insertCompany: CompanyRegisterTO,
  ): Promise<CompanyFullTO | null> {
    this.LOGGER.log(`Creating company for user: ${userId}`);
    const newCompany = await this.db.conn.transaction(async (tx) => {
      var company = await this.findByDNI(insertCompany.company.dni);
      if (!company) {
        this.LOGGER.log(`Saving new company dni: ${insertCompany.company.dni}`);
        company = await this.companyRepository.save(insertCompany.company, tx);
      }

      if (!company) {
        this.LOGGER.error('Company not found');
        throw new Error('Company not found');
      }
      const companyId = company.id;
      const listSubdivisioneDB =
        await this.getSubdivisionsByCompanyId(company.id);
      listSubdivisioneDB.length > 0 &&
        this.LOGGER.log(`Adding more Subdivisions to company: ${company.id}`);

      if ((insertCompany?.listSubdivisions?.length ?? 0) > 0) {
        this.LOGGER.log(
          `Saving company subdivisions for company: ${company.id}`,
        );

        // Filter out subdivisions that already exist in listSubdivisioneDB
        const newSubdivisions = insertCompany.listSubdivisions.filter(
          (subdivision) =>
            !listSubdivisioneDB.some(
              (existing) => existing.id === subdivision.id,
            ) && subdivision.id !== 7777,
        );

        if (newSubdivisions.length > 0) {
          await this.companyRepository.saveCompanySubdivisions(
            company.id,
            newSubdivisions,
            tx,
          );
        }
      }


      if (insertCompany.listKeywords.length > 0) {
        await this.keywordService.saveAll(
          insertCompany.listKeywords.map((keyword) => ({
            ...keyword,
            companyId: +companyId,
            metadata: undefined,
          })),
          tx,
        );
        this.LOGGER.log(`Saving keywords for company: ${companyId}`);
      
        const listKeywordsDB = await this.keywordService.findByCodeCategoriaMercadoPublico(
          insertCompany.listKeywords.map((keyword) => keyword.codeCategoriaMercadoPublico),
          tx,
        );
        this.LOGGER.log(`Deleting old company keywords for company: ${companyId}`);
        await this.companyKeywordService.deleteByCompanyId(companyId, tx);
        this.LOGGER.log(`Saving new company keywords for company: ${companyId}`);
        await this.companyKeywordService.saveAll(
          companyId,
          listKeywordsDB.map((keyword) => keyword.id),
          tx,
        );
      }

      this.LOGGER.log(
        `Saving user-company association for user: ${userId} and company: ${company.id}`,
      );
      await this.userCompanyService.saveUserCompany(userId, company.id, tx);

      this.LOGGER.log(`Company created successfully: ${company.id}`);
      return company;
    });
    return this.findByIdWithFetch(newCompany.id);
  }

  async findByDNI(dni: string): Promise<Company | null> {
    this.LOGGER.log(`Finding company by DNI: ${dni}`);
    return await this.companyRepository.findByDNI(dni);
  }

  async findById(id: number): Promise<Company | null> {
    this.LOGGER.log(`Finding company by id: ${id}`);
    return await this.fwkCacheService.getOrSet<Company | null>(
      FwkCacheServiceImpl.createKey(CacheManagerEnum.FIND_COMPANY_BY_ID, [id]),
      async () => {
        return this.companyRepository.findById(id)
      },
    );
  }

  async findByIdWithFetch(id: number): Promise<CompanyFullTO | null> {
    this.LOGGER.log(`Finding company with fetch by id: ${id}`);
    const company = await this.findById(id);

    if (!company) {
      this.LOGGER.warn(`Company not found by id: ${id}`);
      return null;
    }

    this.LOGGER.log(`Fetching keywords for company: ${company.id}`);
    const listKeywords = await this.companyKeywordService.findByCompanyId(
      company.id,
    );
    this.LOGGER.log(`Fetching subdivisions for company: ${company.id}`);
    const listSubdivisions =
      await this.companyRepository.getSubdivisionsByCompanyId(company.id);

    this.LOGGER.log(`Company with fetch found successfully: ${company.id}`);
    return new CompanyFullTO(
      company,
      listKeywords.map((keyword) => {
        return {
          id: keyword.id,
          value: keyword.value,
          codeCategoriaMercadoPublico: keyword.codeCategoriaMercadoPublico,
          origin: keyword.origin,
          isSelected: true,
          metadata: keyword.metadata,
        };
      }),
      listSubdivisions,
    );
  }
}
