import { Inject, Injectable } from '@nestjs/common';
import { PrimeLogger, TxType } from 'src/framework';
import { KeywordRepository, KeywordService } from 'src/licitaapp/application';
import { MercadoPublicoGirosRepository } from 'src/licitaapp/application/mercado-publico/mp-giros/mercado-publico-giros.repository';
import {
  InsertKeyword,
  Keyword,
  KeywordItem,
  KeywordOrigin,
  Metadata,
} from 'src/licitaapp/domain';

@Injectable()
export class KeywordServiceImpl implements KeywordService {
  private readonly LOGGER = new PrimeLogger(KeywordServiceImpl.name);
  constructor(
    @Inject('KeywordRepository')
    private readonly keywordRepository: KeywordRepository,
    private readonly mpGiros: MercadoPublicoGirosRepository,
  ) {}
  findByCodeCategoriaMercadoPublico(codeCategoriaMercadoPublico: number[], tx?: TxType | undefined): Promise<Keyword[]> {
    this.LOGGER.log(`findByCodeCategoriaMercadoPublico codeCategoriaMercadoPublico: ${JSON.stringify(codeCategoriaMercadoPublico)}`);
    return this.keywordRepository.findByCodeCategoriaMercadoPublico(codeCategoriaMercadoPublico, tx);
  }
  findKeywordsByNameList(names: string[]): Promise<number[]> {
    this.LOGGER.log(`findKeywordsByNameList names: ${JSON.stringify(names)}`);
    return this.keywordRepository.findKeywordsByNameList(names);
  }
  findAdminValues(): Promise<string[]> {
    this.LOGGER.log(`findAdminValues`);
    return this.keywordRepository.findAdminValues();
  }

  updateMetadata(keywordId: number, metadata: Metadata) {
    this.LOGGER.log(
      `updateMetadata keywordId: ${keywordId} metadata: ${JSON.stringify(metadata)}`,
    );
    return this.keywordRepository.updateMetadata(keywordId, metadata);
  }

  getAllWithouthMetadata(tx?: TxType) {
    this.LOGGER.log(`getAllWithouthMetadata`);
    return this.keywordRepository.getAllWithouthMetadata(tx);
  }

  save(keyword: InsertKeyword, tx?: TxType) {
    this.LOGGER.log(`save keyword: ${JSON.stringify(keyword)}`);
    return this.keywordRepository.save(keyword, tx);
  }

  async saveAll(keywords: InsertKeyword[], tx?: TxType) {
    this.LOGGER.log(`saveAll keywords: ${JSON.stringify(keywords)}`);
    const keywordsExisting = await this.keywordRepository.findByCodeCategoriaMercadoPublico(
      keywords.map((k) => k.codeCategoriaMercadoPublico),
    );
    const filteredKeywords = keywords.filter(
      (keyword) => {
        return !keywordsExisting.find((kE) => kE.codeCategoriaMercadoPublico === keyword.codeCategoriaMercadoPublico);
      },
    );
    return this.keywordRepository.saveAll(filteredKeywords, tx);
  }

  async findByName(search: string): Promise<KeywordItem[]> {
    this.LOGGER.log(`findByName keyword: ${search}`);
    if (search.length < 3) {
      return [];
    }
    return this.mpGiros
      .buscarGiros({ q: search })
      .then((response) => {
        if (!response || response.length === 0) {
          return [{ value: search, origin: KeywordOrigin.USER, codeCategoriaMercadoPublico: 0 }];
        } else {
          return response.map((item) => ({
            value: item.text,
            origin: KeywordOrigin.MERCADO_PUBLICO,
            codeCategoriaMercadoPublico: +item.id,
          }));
        }
      })
      .catch(() => {
        return [{ value: search, origin: KeywordOrigin.USER, codeCategoriaMercadoPublico: 0 }];
      });
  }
  async logicalRemove(keywordsId: number) {
    this.LOGGER.log(`logicalRemove keyword: ${keywordsId}`);
    await this.keywordRepository.logicalRemove(keywordsId);
  }
}
