import { Inject, Injectable } from '@nestjs/common';
import { CacheManagerEnum, FwkCacheService, FwkCacheServiceImpl, GeoService } from 'src/framework';
import { PrimeLogger } from 'src/framework/infrastructure/definition/logger/app.exception.logger';
import { MatchingWordRepository } from 'src/licitaapp/application/repository/matching_word-repository/matching_word-repository.interface';
import { MatchingWordService } from 'src/licitaapp/application/service/matching_word-service/matching_word-service.interface';
import { InsertMatchingWord, MatchingWord, UpdateMatchingWord } from 'src/licitaapp/domain';

@Injectable()
export class MatchingWordServiceImpl implements MatchingWordService {
    private readonly LOGGER = new PrimeLogger(MatchingWordServiceImpl.name);
    constructor(
        @Inject('MatchingWordRepository')
        private readonly matchingWordRepository: MatchingWordRepository,
        @Inject('FwkCacheService')
        private readonly fwkCacheService: FwkCacheService,
        @Inject('GeoService')
        private readonly geoService: GeoService,
    ) {}
    getPagination(page: number, pageSize: number, subdivisionId?: string, wordSearch?: string): Promise<MatchingWord[]> {
        this.LOGGER.log(`Getting paginated matching words - page: ${page}, pageSize: ${pageSize}, subdivisionId: ${subdivisionId}, wordSearch: ${wordSearch}`);
        return this.matchingWordRepository.getPagination(page, pageSize, subdivisionId, wordSearch);
    }  

    async findSubdivisionIdByMatchingWords(word: string): Promise<number | null | undefined> {
        if(!word || word.trim() === '') {
            return null;
        }
        this.LOGGER.log(`Finding subdivision id by matching word: ${word}`);  
        const normalizeText = (text: string): string =>
            text
                .toLowerCase()
                .normalize('NFD')
                .replace(/[\u0300-\u036f]/g, '')
                .replace(/[^a-z0-9 ]/g, '')
                .replace('de', '')
                .replace(' y ', '');
        const records = await this.findAll();
        let output = null;
        for (const objectRecord of records) {
            for (const wordItem of objectRecord.coincidence.words) {
                if (word === wordItem || 
                    normalizeText(word).includes(normalizeText(wordItem)) ||
                     normalizeText(wordItem).includes(normalizeText(word))) {
                    output = objectRecord.subdivisionId;
                    break;
                }
                if (output) break;
            }
        }

        const infoSubdivision = await this.geoService.searchSubdivisionsByName([word]);
        if(infoSubdivision) {
            output = infoSubdivision.subdivision.id;
        }
        return output;
    }      
    async findAll(): Promise<MatchingWord[]> {
        this.LOGGER.log('Finding all matching words');
        return await this.fwkCacheService.getOrSet<MatchingWord[]>(
            CacheManagerEnum.FIND_ALL_MATCHING_WORD,
            async () => {
                this.LOGGER.log(`Fetching matching words from DB`);
                return this.matchingWordRepository.findAll()
            },
        );
    }
    async findById(id: number): Promise<MatchingWord | null | undefined> {
        this.LOGGER.log(`Finding matching word by id: ${id}`);
        return await this.fwkCacheService.getOrSet<MatchingWord | null | undefined>(
           FwkCacheServiceImpl.createKey(CacheManagerEnum.FIND_ONE_MATCHING_WORD, [id]),
            async () => {
                return this.matchingWordRepository.findById(id)
            },
        );
    }
    async save(matchingWord: InsertMatchingWord): Promise<boolean> {
        this.LOGGER.log(`Saving matching word: ${JSON.stringify(matchingWord)}`);
        return await this.matchingWordRepository.save(matchingWord).then(result => {
            this.fwkCacheService.del(CacheManagerEnum.FIND_ALL_MATCHING_WORD);
            return result;
        });
    }
    async update(matchingWord: UpdateMatchingWord): Promise<boolean> {
        this.LOGGER.log(`Updating matching word with id: ${matchingWord.id}`);
        return await this.matchingWordRepository.update(matchingWord).then(result => {
            this.fwkCacheService.del(CacheManagerEnum.FIND_ALL_MATCHING_WORD);
            return result;
        });
    }
    delete(matchingWordId: number): Promise<boolean> {
        this.LOGGER.log(`Deleting matching word with id: ${matchingWordId}`);
        return this.matchingWordRepository.delete(matchingWordId).then(result => {
            this.fwkCacheService.del(CacheManagerEnum.FIND_ALL_MATCHING_WORD);
            return result;
        });
    }
}
