import { Inject, Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { Cron, CronExpression } from '@nestjs/schedule';
import {
  FirebaseService,
  FwkCacheService,
  InsertNotificationRecord,
  PrimeLogger,
  UserCampaignsService,
  UserRequestService,
  UserRoleService,
} from 'src/framework';
import { NotificationRecordService } from 'src/framework/application/service/notification-record-service/notification-record-service.interface';
import { CompanyService, UserCompanyService } from 'src/licitaapp/application';
import { ScheduleMpService } from 'src/licitaapp/application/service/schedule-mp-service/schedule-mp-service.interface';
import { TenderFetcherService } from 'src/licitaapp/application/service/tender-fetcher-service/tender-fetcher-service.interface';
import { TenderService } from 'src/licitaapp/application/service/tender-service/tender-service.interface';
import { UserCompanyAgileTenderService } from 'src/licitaapp/application/service/user-company-agile-tender-service/user-company-agile-tender-service.interface';
import { ApplicationTypeEnum } from 'src/licitaapp/domain/enum/enum.definition';
import { TenderUtil } from 'src/licitaapp/domain/util';

@Injectable()
export class ScheduleMpServiceImpl implements ScheduleMpService {
  private readonly LOGGER = new PrimeLogger(ScheduleMpServiceImpl.name);
  private readonly isCronActive: boolean;
  private hourToSearchTenders = [4];
  private perFiveMinutes = [5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55];
  private allHours = [
    6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23,
  ];
  constructor(
    private readonly configService: ConfigService,
    @Inject('UserCompanyAgileTenderService')
    private readonly userCompanyAgileTenderService: UserCompanyAgileTenderService,
    @Inject('TenderService') private readonly tenderService: TenderService,
    @Inject('TenderFetcherService')
    private readonly tenderFetcherService: TenderFetcherService,
    @Inject('UserCampaignsService')
    private readonly userCampaignsService: UserCampaignsService,
    @Inject('NotificationRecordService')
    private readonly notificationRecordService: NotificationRecordService,
    @Inject('UserCompanyService')
    private readonly userCompanyService: UserCompanyService,
    @Inject('FwkCacheService')
    private readonly fwkCacheService: FwkCacheService,
    @Inject('FirebaseService')
    private readonly firebaseService: FirebaseService,
    @Inject('UserRequestService')
    private readonly userRequestService: UserRequestService,
    @Inject('CompanyService') private readonly companyService: CompanyService,
    @Inject('UserRoleService') private readonly userRoleService: UserRoleService,
  ) {
    this.isCronActive =
      this.configService
        .get<string>('CRON_JOB_ENABLED', 'true')
        .toLowerCase() === 'true';
  }

  // dejarlo cada 10 minutos.
  @Cron(CronExpression.EVERY_DAY_AT_10PM, {
    name: 'sendNotificationsRecords',
    timeZone: TenderUtil.TIMEZONE,
  })
  async sendNotificationsRecords() {

    if (!this.isCronActive) {
      this.LOGGER.warn('sendNotificationsRecords - Cron job is disabled');
      return;
    }

    const currentTime = TenderUtil.getCurrentSystemDate();
    const hour = currentTime.getHours();
    const minute = currentTime.getMinutes();
    this.LOGGER.log('sendNotificationsRecords - START');
    try {
      await this.firebaseService.sendAllNotificationsRecords();

      // cada hora al minuto 17 (entre las 6 y las 23)
      if (this.allHours.includes(hour) && minute === 30) {
        const countActiveRequests =
          await this.userRequestService.countActiveUserRequests();
        countActiveRequests > 0 && await this.firebaseService.notifyToAdminUsers(countActiveRequests);
      }
    } catch (error) {
      this.LOGGER.error(`sendNotificationsRecords - Error: ${error}`);
    }
  }

  @Cron(CronExpression.EVERY_MINUTE, {
       name: 'mainCentralScheduleMp',
       timeZone: TenderUtil.TIMEZONE,
    })
  async mainCentralScheduleMp() {

    if (!this.isCronActive) {
      this.LOGGER.warn('mainCentralScheduleMp - Cron job is disabled');
      return;
    }

    const currentTime = TenderUtil.getCurrentSystemDate();
    this.LOGGER.log(
      `mainCentralScheduleMp - currentTime: ${currentTime.toLocaleString('es-CL', { timeZone: TenderUtil.TIMEZONE })}`,
    );
    // task per minute

    this.tenderService.tenderTaskPerMinutes();

    const hour = currentTime.getHours();
    const minute = currentTime.getMinutes();
    this.LOGGER.log(`mainCentralScheduleMp - hour: ${hour}, minute: ${minute}`);

    // todas las 4Am
    if (this.hourToSearchTenders.includes(hour) && minute === 0) {
      let applicationLogId = await this.tenderService.createApplicationLogID(
        ApplicationTypeEnum.TENDERS_OF_THE_DAY_LOG,
      );
      await this.tenderFetcherService.logAndExecute(
        ApplicationTypeEnum.TENDERS_OF_THE_DAY_LOG,
        applicationLogId,
        this.tenderFetcherService.callfetchTendersOfTheDay.bind(
          this.tenderFetcherService,
        ),
      );
      applicationLogId = await this.tenderService.createApplicationLogID(
        ApplicationTypeEnum.FETCH_TENDER_AGILE_LOG,
      );
      await this.tenderFetcherService.logAndExecute(
        ApplicationTypeEnum.FETCH_TENDER_AGILE_LOG,
        applicationLogId,
        this.tenderFetcherService.callFetchAgileTenders.bind(
          this.tenderFetcherService,
        ),
      );
      applicationLogId = await this.tenderService.createApplicationLogID(
        ApplicationTypeEnum.FETCH_JOIN_AGILE_TENDER_LOG,
      );
      await this.tenderFetcherService.logAndExecute(
        ApplicationTypeEnum.FETCH_JOIN_AGILE_TENDER_LOG,
        applicationLogId,
        this.tenderFetcherService.joinAgileTenderWithKeywords.bind(
          this.tenderFetcherService,
        ),
      );     
    }

    // task per 5 minutes
    if (this.perFiveMinutes.includes(minute)) {
      this.tenderService.tenderTaskEvery5Minutes();
    }

    // 12:22 de la mañana se ejecuta una vez al día
    if (hour === 0 && minute === 22) {
      this.LOGGER.log('mainCentralScheduleMp - Executing daily tasks');
      this.tenderService.tenderTaskPerDay();

      this.checkFewKeywordsToCompany();
    }

    // cada 2 horas en horas impares con 7 minutos
    if (hour % 2 !== 0 && minute === 7) {
      this.LOGGER.log('mainCentralScheduleMp - Executing tasks every 2 hours');
      this.tenderService.tenderTaskBy2Hour();
    }

    // cada 2 horas en horas pares con 18 minutos
    if (hour % 2 == 0 && minute === 18) {
      this.LOGGER.log(
        'mainCentralScheduleMp - Executing tasks every 2 hours pairs',
      );
      const { end } = TenderUtil.getDayStartAndEnd(
        TenderUtil.getCurrentSystemDate(),
      );
      const listTenderAgileIds =
        await this.tenderService.findByCloseDateAgileTender(
          TenderUtil.subtractDays(end, 1),
        );
      this.userCompanyAgileTenderService.erraseUserCompanyAgileTender(
        listTenderAgileIds,
      );
      this.tenderService.erraseAgileOldTenders(listTenderAgileIds);
      await this.fwkCacheService.del();
      
      /*applicationLogId = await this.tenderService.createApplicationLogID(
        ApplicationTypeEnum.METADATA_AGILE_TENDERS_LOG,
      );
      await this.tenderFetcherService.logAndExecute(
        ApplicationTypeEnum.METADATA_AGILE_TENDERS_LOG,
        applicationLogId,
        this.tenderService.checkMetadataAgileTenders.bind(
          this.tenderService,
        ),
      );*/
      
      /// this.tenderFetcherService.getAgileTenders();
    }

    // cada hora al minuto 17 (entre las 6 y las 23)
    if (this.allHours.includes(hour) && minute === 17) {
      this.LOGGER.log('mainCentralScheduleMp - Executing tasks every hour');
      this.tenderService.tenderTaskByHour();
      //this.tenderFetcherService.getAgileTenders();
    }

    this.LOGGER.log(
      `mainCentralScheduleMp - Running CAMPAIGN at ${hour}:${minute}`,
    );
    this.createCampaign(hour, minute);
  }

  // revisa si hay usuarios con pocos rubros.
  async checkFewKeywordsToCompany(): Promise<void> {
    this.LOGGER.log('checkFewKeywordsToCompany - START');
    const companiesToNotify =
      await this.companyService.getAllCompanyWithoutKeywords();

    const adminIds = await this.userRoleService.getAdminUserIds();

    const notificationsArray: InsertNotificationRecord[] = [];
    for (const companyId of companiesToNotify) {
      const companyData = await this.companyService.findById(companyId);
      const listUser =
        await this.userCompanyService.getInfoUserCompany(companyId);

      for (const userData of listUser) {
        this.LOGGER.log(
          `createCampaign - Creating notification for user with id: ${userData.userId}`,
        );
        notificationsArray.push(
          {
            userId: userData.userId,
            title: `Añade más Rubros`,
            defaultMessage: 'Tienes muy pocos Rubros asociados a tu empresa...',
            active: true,
          }
        );
      }

      // notifica a los admins
      adminIds.forEach((adminId) => {
      notificationsArray.push({
        userId: adminId,
        title: `Asistencia de Rubros`,
            defaultMessage: `Admin: La empresa ${companyData?.socialReason} tiene 1 Rubro asociado.`,
        active: true,
        isAdmin: true,
      });
    });
    }

    notificationsArray.length > 0 &&
      (await this.notificationRecordService.saveAll(notificationsArray));
    this.LOGGER.log('checkFewKeywordsToCompany - END');
  }

  async createCampaign(hour: number, minute: number): Promise<void> {
    const campaings = await this.userCampaignsService.getActiveCampaigns();

    if (!campaings || campaings.length === 0) {
      this.LOGGER.warn('createCampaign - No active campaigns found');
      return;
    }
    this.LOGGER.log(
      `createCampaign - Found ${campaings.length} active campaigns`,
    );
    for (const campaign of campaings) {
      const daysOfWeek = campaign.dayOfWeek.split('-').map(Number);
      if (!daysOfWeek.includes(TenderUtil.getCurrentSystemDate().getDay())) {
        this.LOGGER.log(
          `createCampaign - Skipping campaign ${campaign.id} for day ${campaign.dayOfWeek}`,
        );
        continue;
      }
      this.LOGGER.log(`createCampaign - Processing campaign ${campaign.id}`);
      if (!campaign.defHour || !campaign.defMinute) {
        this.LOGGER.warn(
          `createCampaign - Campaign ${campaign.id} does not have a valid time set`,
        );
        continue;
      } else {
        this.LOGGER.warn(
          `Campaign ${campaign.id} has time set to ${campaign.defHour}:${campaign.defMinute} compare with ${hour}:${minute}`,
        );
        if (campaign.defHour == hour && campaign.defMinute == minute) {
          this.LOGGER.log(
            `createCampaign - Creating campaign for user with id: ${campaign.id}`,
          );
          const activeUserIds =
            await this.userCompanyService.getUserIdsWithActiveCompany();
          const notificationsArray: InsertNotificationRecord[] = [];
          for (const userID of activeUserIds) {
            this.LOGGER.log(
              `createCampaign - Creating notification for user with id: ${userID}`,
            );
            notificationsArray.push({
              userId: userID,
              title: campaign.title,
              defaultMessage: campaign.defaultMessage,
              active: true,
            });
          }
          notificationsArray.length > 0 &&
            (await this.notificationRecordService.saveAll(notificationsArray));
        }
      }
    }
  }
}
