import { forwardRef, Inject, Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import * as moment from 'moment-timezone';
import { FwkCacheService, PrimeLogger } from 'src/framework';
import { AgileTenderFullInfoTO, AgileTenderPageResponse, AgileTenderTO, CompanyService, DetailAgileTenderTO, LicitacionesQueryParams, LicitacionesResponse, UserCompanyService } from 'src/licitaapp/application';
import { MercadoPublicoRepository } from 'src/licitaapp/application/mercado-publico/mp.repository';
import { ApplicationLogService } from 'src/licitaapp/application/service/application-log-service/application-log-service.interface';
import { CompanyKeywordService } from 'src/licitaapp/application/service/company-keyword-service/company-keyword-service.interface';
import { MatchingWordService } from 'src/licitaapp/application/service/matching_word-service/matching_word-service.interface';
import { TenderFetcherService } from 'src/licitaapp/application/service/tender-fetcher-service/tender-fetcher-service.interface';
import { TenderKeywordCategoryService } from 'src/licitaapp/application/service/tender-keyword-category-service/tender-keyword-category-service.interface';
import { TenderService } from 'src/licitaapp/application/service/tender-service/tender-service.interface';
import { UserCompanyAgileTenderService } from 'src/licitaapp/application/service/user-company-agile-tender-service/user-company-agile-tender-service.interface';
import { InsertAgileTender, Licitacion } from 'src/licitaapp/domain';
import { ApplicationTypeEnum } from 'src/licitaapp/domain/enum/enum.definition';
import { TenderUtil } from 'src/licitaapp/domain/util';
import { inspect } from 'util';

@Injectable()
export class TenderFetcherServiceImpl implements TenderFetcherService {
    private readonly LOGGER = new PrimeLogger(TenderFetcherServiceImpl.name);
    static readonly TIMEZONE = 'America/Santiago';
    private readonly maxRetries;
    private readonly retryInterval;
    private readonly fetchFromDaysAgo;
    private readonly enabled;

    constructor(
      private readonly configService: ConfigService,
      @Inject(forwardRef(() => 'MercadoPublicoRepository')) private readonly mercadoPublicoAPI: MercadoPublicoRepository,
      @Inject(forwardRef(() => 'TenderService')) private readonly tenderService: TenderService,
      @Inject(forwardRef(() => 'UserCompanyAgileTenderService')) private readonly userCompanyAgileTenderService: UserCompanyAgileTenderService,
      @Inject(forwardRef(() => 'ApplicationLogService')) private readonly applicationLogService: ApplicationLogService,
      @Inject(forwardRef(() => 'MatchingWordService')) private readonly matchingWordService: MatchingWordService,
      @Inject(forwardRef(() => 'TenderKeywordCategoryService')) private readonly tenderKeywordCategoryService: TenderKeywordCategoryService,
      @Inject(forwardRef(() => 'CompanyService')) private readonly companyService: CompanyService,
      @Inject(forwardRef(() => 'UserCompanyService')) private readonly userCompanyService: UserCompanyService,
      @Inject(forwardRef(() => 'CompanyKeywordService')) private readonly companyKeywordService: CompanyKeywordService,
      @Inject(forwardRef(() => 'FwkCacheService'))
      private readonly fwkCacheService: FwkCacheService,
    ) {
      this.maxRetries = this.configService.get<number>(
        'TENDER_FETCHER_MAX_RETRIES',
        2,
      );
      this.retryInterval = this.configService.get<number>(
        'TENDER_FETCHER_RETRY_INTERVAL',
        1000,
      );
      this.fetchFromDaysAgo = this.configService.get<number>(
        'TENDER_FETCHER_FETCH_FROM_DAYS_AGO',
        90,
      );
      this.enabled =
        this.configService.get<'true' | 'false'>(
          'TENDER_FETCHER_ENABLED',
          'false',
        ) === 'true';
    }
  async joinAgileTenderWithKeywords(applicationLogId: number, companyId?: number, userID?: number): Promise<string> {
    var messageTxt = `ApplicationLogId: ${applicationLogId} - inicio`;

    const companyIds = companyId ? [companyId] : await this.companyService.getCompanyIdsActive();    
    
    for (const currentCompanyId of companyIds) {
      messageTxt += ` Procesando companyId: ${currentCompanyId}.`;
      const keywordsCompany = await this.companyKeywordService.findByCompanyId(currentCompanyId);
      messageTxt += ` Encontradas ${keywordsCompany.length} palabras clave.`;
      
      for (const keyword of keywordsCompany) {
        messageTxt += ` Procesando keyword: ${keyword.codeCategoriaMercadoPublico}.`;
        const resultMatchAgileTenderCompany = await this.mercadoPublicoAPI.matchAgileKeyword(keyword.codeCategoriaMercadoPublico);
        messageTxt += ` Encontrados ${resultMatchAgileTenderCompany.payload ? resultMatchAgileTenderCompany.payload.pageSize * resultMatchAgileTenderCompany.payload.pageCount : 0} licitaciones ágil para la palabra clave: ${keyword.codeCategoriaMercadoPublico}.`;
        if (resultMatchAgileTenderCompany.payload && resultMatchAgileTenderCompany.payload.pageSize > 0) {
          for (const tender of resultMatchAgileTenderCompany.payload.resultados) {
            messageTxt += ` Asociando licitación ágil código: ${tender.codigo} - company ${currentCompanyId} - rubroMatch: ${keyword.id}.`;
            const fullDetail = await this.mercadoPublicoAPI.licitacionAgilDetail(tender.codigo);
            const tenderAgileId = await this.saveAgileTender({
              general: tender,
              licitacionAgilDetail: fullDetail,
            });
            for(const informacion of fullDetail?.productos_solicitados || []) {
              await this.tenderKeywordCategoryService.saveAgileTenderCategory(
                tenderAgileId,
                `${keyword.codeCategoriaMercadoPublico}`,
                informacion.descripcion
              );  
            } 

            if(userID){
              this.userCompanyAgileTenderService.saveAll(
                userID,
                currentCompanyId,
                [tenderAgileId],
                false,
              );
            }else{
              const userInfo = await this.userCompanyService.getInfoUserCompany(currentCompanyId);
              for(const user of userInfo){
                this.userCompanyAgileTenderService.saveAll(
                  user.userId,
                  currentCompanyId,
                  [tenderAgileId],
                  false,
                );
              }              
            }
            
          }
        }
      }
    }
    this.LOGGER.log(messageTxt);
    return Promise.resolve(messageTxt);
  }
  
  async callFetchAgileTenders(applicationLogId: number): Promise<string> {
    var messageTxt = `ApplicationLogId: ${applicationLogId} - inicio`;
    messageTxt += await this.getAgileTenders();
    return Promise.resolve(messageTxt);
  }
  async logAndExecute(type: ApplicationTypeEnum, applicationLogId: number, action: (applicationLogId: number) => Promise<string>): Promise<void> {
    this.LOGGER.log(`logAndExecute - type: ${type}, applicationLogId: ${applicationLogId}`);
    const startTime = Date.now();
    try {
      const message = await action(applicationLogId);
      const duration = Date.now() - startTime;
      await this.applicationLogService.updateState(applicationLogId, 10, `Duración: ${duration/1000} s. ${message}.`);
    } catch (error) {
      const duration = Date.now() - startTime;
      this.LOGGER.error(`logAndExecute - ${type} - Error: ${error}`);
      await this.applicationLogService.updateState(applicationLogId, 9, TenderUtil.evalTextLarge(`Duración: ${duration/1000} s. Error: ${inspect(error)}.`));
    }
  }

    async fetchTendersWithRetry(args: LicitacionesQueryParams): Promise<LicitacionesResponse | undefined> {
        const delay = (ms: number) =>
        new Promise((resolve) => setTimeout(resolve, ms));

        for (let attempt = 1; attempt <= this.maxRetries; attempt++) {
            try {
                return await this.mercadoPublicoAPI.licitaciones(args);
            } catch (error) {
                //if (attempt === Number(this.maxRetries)) {
                throw error;
            //}
            await delay(attempt * this.retryInterval); // Exponential backoff
            }
        }
    }
    async processTenders(args: LicitacionesQueryParams): Promise<boolean> {
      this.LOGGER.debug(`Fetching tenders with args: ${JSON.stringify(args)}`);
      const startTime = Date.now();
      let messageTxt = 'Iniciado a las ' + TenderUtil.getCurrentSystemDate().toLocaleTimeString();
      const applicationLogId = await this.applicationLogService.save({
        userName: 'SYSTEM',
        statusTypeId: 8,
        detail: messageTxt,
        type: ApplicationTypeEnum.SEARCH_TENDERS_MERCADO_PUBLICO_LOG,
      });

      try {
        const tenders = await this.mercadoPublicoAPI.licitacionesListadoDiarioGeneral(args.fecha!);
        messageTxt += ` - Licitaciones encontradas: ${tenders?.Cantidad || 0}`;
        if (tenders && tenders.Cantidad > 0) {
          let processed = 0;
          this.LOGGER.debug(`Fetched ${tenders.Cantidad} tenders. Processing...`);

          for (const tender of tenders.Listado) {
            processed++;
            this.LOGGER.debug(
              `Tender: ${tender.Nombre}, codigo ${tender.CodigoExterno}`
            );
            messageTxt += ` - Procesando licitación código: ${tender.CodigoExterno} (${processed}/${tenders.Cantidad})`;
            try {
              const codeDetails = await this.mercadoPublicoAPI.licitacionesDetalleGeneral(tender.CodigoExterno);

              if (codeDetails?.Listado) {
                for (const tenderDetails of codeDetails.Listado) {
                  
                  const trace = `Upserting tender: ${tenderDetails.CodigoExterno} processed ${processed}/${tenders.Cantidad} tiempo transcurrido: ${(Date.now() - startTime) / 1000} s.`;
                  this.LOGGER.debug(
                    trace
                  );
                  await this.applicationLogService.updateState(applicationLogId, 8, `${trace}. Trabajando...`);
                  const idTender = await this.tenderService.upsert({
                    code: tenderDetails.CodigoExterno,
                    name: tenderDetails.Nombre,
                    description: tenderDetails.Descripcion,
                    details: tenderDetails,
                    isFavorite: false,
                  });
                  const codesKeywords = this.getCodesFromItems(tenderDetails);
                  for(const codeKeyword of codesKeywords){
                    await this.tenderKeywordCategoryService.saveNormalTenderCategory(
                      idTender,
                      codeKeyword,
                      tenderDetails.Descripcion 
                    );
                  }
                  
                }
              }
            } catch (error) {
              const duration = Date.now() - startTime;
              messageTxt += `Duración: ${duration/1000} s. Error: ${inspect(error)}.`
              await this.applicationLogService.updateState(applicationLogId, 9, messageTxt);
              this.LOGGER.error(
                `Error processing tenders for code ${tender.CodigoExterno} processed ${processed}/${tenders.Cantidad}`
              );
              continue;
            }
          }
          messageTxt += `procesados ${processed}/${tenders.Cantidad} tiempo transcurrido: ${(Date.now() - startTime) / 1000} s.`
          await this.applicationLogService.updateState(applicationLogId, 10, messageTxt);
        } else {
          messageTxt += ' - No se encontraron licitaciones.';
          this.LOGGER.debug(
            `No tenders found for args: ${JSON.stringify(args)}`
          );
        }
      } catch (error) {
        const duration = Date.now() - startTime;
        messageTxt += `Duración: ${duration/1000} s. Error: ${inspect(error)}.`;
        this.applicationLogService.updateState(applicationLogId, 9, messageTxt);
        this.LOGGER.error(
          `Error processing tenders for args: ${JSON.stringify(args)}`
        );
        return false;
      }
      this.LOGGER.debug('Finished processing tenders');
      return true;
    }

    private getCodesFromItems(tenderDetails: Licitacion): string[] {
      return tenderDetails.Items.Listado.map(item => {
        return item.CodigoCategoria;
      })
    }

    async callfetchTendersOfTheDay(applicationLogId: number): Promise<string>{
      var messageTxt = `ApplicationLogId: ${applicationLogId} - inicio`;
      await this.fetchTendersOfTheDay();
      return messageTxt;
    }
    // licitaciones normales busqueda diaria
    async fetchTendersOfTheDay(ddmmyyyy?: string): Promise<boolean | undefined> {
        if (!this.enabled) {
            this.LOGGER.log('Tender fetcher is disabled, skipping daily fetch.');
            return;
        }
        const now = moment().tz(TenderFetcherServiceImpl.TIMEZONE);
        const today = now.format('DDMMYYYY');
        const todayInWords = now.format('dddd, MMMM Do YYYY');
        this.LOGGER.debug(
        `Fetching tenders of the day ${todayInWords} (${TenderFetcherServiceImpl.TIMEZONE})...`
        );
        return await this.processTenders({ fecha: ddmmyyyy || today });
    }
    private async getAgileTenders(): Promise<string> {
      if (!this.enabled) {
          this.LOGGER.log('Tender fetcher is disabled, skipping agile tenders fetch.');
          return 'Tender fetcher is disabled, skipping agile tenders fetch.';
      }
      const now = moment().tz(TenderFetcherServiceImpl.TIMEZONE);
      const tomorrow = now.add(1, 'day').format('YYYY-MM-DD');
      const lastMonth = moment().tz(TenderFetcherServiceImpl.TIMEZONE).subtract(30, 'days').format('YYYY-MM-DD');
      const todayInWords = now.format('dddd, MMMM Do YYYY');
      this.LOGGER.debug(
       `Fetching tenders of the day ${todayInWords} (${TenderFetcherServiceImpl.TIMEZONE})...`
      );
      let messageTxt = `Fetching agile tenders from ${lastMonth} to ${tomorrow} (${TenderFetcherServiceImpl.TIMEZONE})...`;
      const firstPageInfo = await this.processAgileTender({ initDate: lastMonth, endDate: tomorrow, page: 1 });
      this.LOGGER.debug(`Saving first page results...`);
      const fullDetail = firstPageInfo && await this.mercadoPublicoAPI.licitacionAgilDetail(firstPageInfo.payload.resultados[0].codigo);
      const tenderAgileId = firstPageInfo && await this.saveAgileTender({
        general: firstPageInfo.payload.resultados[0],
        licitacionAgilDetail: fullDetail,
      });
      if(tenderAgileId){
        for(const informacion of fullDetail?.productos_solicitados || []) {
          await this.tenderKeywordCategoryService.saveAgileTenderCategory(
            tenderAgileId,
            `${informacion.codigo_producto}`,
            informacion.descripcion
          );  
        }  
      }      

      if(!firstPageInfo) {
        this.LOGGER.debug('No agile tenders found for today.');
        return 'No agile tenders found for today.';
      }
      messageTxt += ` Found ${firstPageInfo.payload.pageCount} pages.`;
      const totalPages = firstPageInfo?.payload.pageCount || 0;
      let resultPage;
      for (let page = 2; page <= totalPages; page++) {
        this.LOGGER.debug(`Processing page ${page} of ${totalPages}`);
        resultPage = await this.processAgileTender({ initDate: lastMonth, endDate: tomorrow, page });
        this.LOGGER.debug(`Saving page ${page} results...`);
        if(!resultPage) {
          this.LOGGER.debug(`No agile tenders found on page ${page}.`);
          continue;
        }
        messageTxt += ` Cargadas ${resultPage.payload.resultados.length} licitaciones.`;
        for (const tender of resultPage.payload.resultados) {
          const fullDetail = await this.mercadoPublicoAPI.licitacionAgilDetail(tender.codigo);
          const tenderAgileIdDB = await this.saveAgileTender({
            general: tender,
            licitacionAgilDetail: fullDetail,
          });

          if(tenderAgileIdDB){
            for(const informacion of fullDetail?.productos_solicitados || []) {
              await this.tenderKeywordCategoryService.saveAgileTenderCategory(
                tenderAgileIdDB,
                `${informacion.codigo_producto}`,
                informacion.descripcion
              );  
            }  
          }
        }
      }
      messageTxt += ` Processed ${totalPages} pages.`;
      await this.fwkCacheService.del();
      return messageTxt;

    }

    public addDetailAgileTender(data: DetailAgileTenderTO | undefined){
      let outText = data ? 'Productos:\n' : '';
      if(data && data.productos_solicitados && data.productos_solicitados.length > 0){
        for(const producto of data.productos_solicitados){
          outText = outText.concat(`- ${producto.nombre}\n`);
        }
      }
      return outText;
    }

    async saveAgileTender(tenderInfo: AgileTenderFullInfoTO): Promise<number> {
      const tender = new InsertAgileTender();
      tender.code = tenderInfo.general.codigo;
      tender.name = tenderInfo.general.nombre;
      tender.description = `Disponible: ${TenderUtil.convertToMillions(tenderInfo.general.monto_disponible)} ${tenderInfo.general.moneda}\n${this.addDetailAgileTender(tenderInfo.licitacionAgilDetail)}`;
      tender.descriptionLocation = tenderInfo.general.organismo ? tenderInfo.general.organismo : '';
      tender.closeDate =  moment.tz(tenderInfo.general.fecha_cierre, TenderFetcherServiceImpl.TIMEZONE).toDate();
      tender.details = tenderInfo;
      tender.createdAt = moment.tz(tenderInfo.general.fecha_publicacion, TenderFetcherServiceImpl.TIMEZONE).toDate();
      tender.subdivisionId = await this.matchingWordService.findSubdivisionIdByMatchingWords(tenderInfo.general.organismo);
      tender.amountCLP = tenderInfo.general.monto_disponible_CLP;
      return await this.tenderService.upsertAgileTender(tender);
    }

    private async processAgileTender(args: { initDate: string; endDate: string; page: number; }): Promise<AgileTenderPageResponse | undefined> {
      this.LOGGER.debug(
        `Fetching agile tenders with args: ${JSON.stringify(args)}`
      );
      const delay = (ms: number) =>
      new Promise((resolve) => setTimeout(resolve, ms));

      for (let attempt = 1; attempt <= 3; attempt++) {
          try {
              return await this.mercadoPublicoAPI.licitacionesAgiles(args);
          } catch (error) {
              if (attempt === 3) {
              //throw error;
              this.LOGGER.error(
                `Error fetching agile tenders with args: ${JSON.stringify(args)} - ${inspect(error)}`
              );
          }
          await delay(attempt * this.retryInterval); // Exponential backoff
          }
      }

    }

    async syncTenders(): Promise<void> {
        if (!this.enabled) {
            return;
        }
        this.LOGGER.debug(`Fetching tenders of the last ${this.fetchFromDaysAgo} days...`);
        const today = moment().tz(TenderFetcherServiceImpl.TIMEZONE);
        for (let i = 0; i < this.fetchFromDaysAgo; i++) {
        const date = today.subtract(i, 'days').format('DDMMYYYY');
        await this.processTenders({ fecha: date });
        }
    }
}
